package com.xebialabs.deployit.jcr;

import org.joda.time.DateTime;

import javax.jcr.*;
import java.util.*;

import static com.xebialabs.overthere.util.OverthereUtils.checkState;


public class JcrUtils {

    public static List<String> readList(Node node, String property) throws RepositoryException {
        Property prop = node.getProperty(property);
        List<String> list = new ArrayList<>();
        if (prop.isMultiple()) {
            for (Value value : prop.getValues()) {
                list.add(value.getString());
            }
        } else {
            list.add(prop.getString());
        }
        return list;
    }

    public static Map<String, String> readMap(Property property) throws RepositoryException {
        Map<String, String> map = new HashMap<>();
        checkState(property.isMultiple(), "The property [%s] should be multi-valued in order to read a Map<String, String>", property.getName());
        for (Value value : property.getValues()) {
            String string = value.getString();
            int i = string.indexOf("=");
            String k = string;
            String v = "";
            if (i > -1) {
                k = string.substring(0, i);
                v = string.substring(i + 1);
            }
            map.put(k, v);
        }
        return map;
    }

    public static void writeMap(Node node, String propertyName, Map<String, String> map) throws RepositoryException {
        List<Value> values = new ArrayList<>();
        for (Map.Entry<String, String> entry : map.entrySet()) {
            values.add(node.getSession().getValueFactory().createValue(entry.getKey() + "=" + entry.getValue()));
        }

        if (values.isEmpty() && node.hasProperty(propertyName)) {
            node.getProperty(propertyName).remove();
        }

        if (!values.isEmpty()) {
            node.setProperty(propertyName, values.toArray(new Value[values.size()]));
        }
    }

    public static void forEachNonJcrProperty(Node node, Callback<Property> f) throws RepositoryException {
        PropertyIterator properties = node.getProperties();
        while (properties.hasNext()) {
            Property p = properties.nextProperty();

            if (p.getName().startsWith("jcr:") || p.getName().startsWith(JcrConstants.METADATA_PROPERTY_NAME_PREFIX))
                continue;
            f.apply(p);
        }
    }

    public static void clearProperties(Node node) throws RepositoryException {
        forEachNonJcrProperty(node, new Callback<Property>() {
            public void apply(Property input) throws RepositoryException {
                input.remove();
            }
        });
    }

    public static String getProperty(Node node, String propName) throws RepositoryException {
        return node.getProperty(propName).getString();
    }

    public static int getIntegerProperty(Node node, String propName) throws RepositoryException {
        return (int) node.getProperty(propName).getLong();
    }

    public static DateTime getDate(Node node, String propName) throws RepositoryException {
        if (node.hasProperty(propName)) {
            return new DateTime(node.getProperty(propName).getDate());
        }
        return null;
    }

    public static String getOptionalProperty(Node node, String propName) throws RepositoryException {
        if (node.hasProperty(propName)) {
            return node.getProperty(propName).getString();
        }
        return null;
    }

    public static Node getOrCreateChild(Node parentNode, String nodeId) throws RepositoryException {
        if (parentNode.hasNode(nodeId)) {
            return parentNode.getNode(nodeId);
        } else {
            return parentNode.addNode(nodeId);
        }
    }

    public static int countMatches(String s, char c) {
        int matches = 0;
        for (int i = 0; i < s.length(); i++) {
            if (s.charAt(i) == c) {
                matches++;
            }
        }
        return matches;
    }

    public interface Callback<T> {
        public void apply(T input) throws RepositoryException;
    }

    /**
     * Escapes all illegal JCR name characters of a string.
     * The encoding is loosely modeled after URI encoding, but only encodes
     * the characters it absolutely needs to in order to make the resulting
     * string a valid JCR name.
     * Use {@link #unescapeIllegalJcrChars(String)} for decoding.
     * <p>
     * QName EBNF:<br>
     * <xmp>
     * simplename ::= onecharsimplename | twocharsimplename | threeormorecharname
     * onecharsimplename ::= (* Any Unicode character except: '.', '/', ':', '[', ']', '*', '|' or any whitespace character *)
     * twocharsimplename ::= '.' onecharsimplename | onecharsimplename '.' | onecharsimplename onecharsimplename
     * threeormorecharname ::= nonspace string nonspace
     * string ::= char | string char
     * char ::= nonspace | ' '
     * nonspace ::= (* Any Unicode character except: '/', ':', '[', ']', '*', '|' or any whitespace character *)
     * </xmp>
     *
     * @param name the name to escape
     * @return the escaped name
     */
    public static String escapeIllegalJcrChars(String name) {
        return escapeIllegalChars(name, "%/:[]*|\t\r\n");
    }

    private static String escapeIllegalChars(String name, String illegal) {
        StringBuilder buffer = new StringBuilder(name.length() * 2);
        for (int i = 0; i < name.length(); i++) {
            char ch = name.charAt(i);
            if (illegal.indexOf(ch) != -1
                    || (ch == '.' && name.length() < 3)
                    || (ch == ' ' && (i == 0 || i == name.length() - 1))) {
                buffer.append('%');
                buffer.append(Character.toUpperCase(Character.forDigit(ch / 16, 16)));
                buffer.append(Character.toUpperCase(Character.forDigit(ch % 16, 16)));
            } else {
                buffer.append(ch);
            }
        }
        return buffer.toString();
    }

    /**
     * Unescapes previously escaped jcr chars.
     * <p>
     * Please note, that this does not exactly the same as the url related
     * {@link #unescape(String)}, since it handles the byte-encoding
     * differently.
     *
     * @param name the name to unescape
     * @return the unescaped name
     */
    public static String unescapeIllegalJcrChars(String name) {
        StringBuilder buffer = new StringBuilder(name.length());
        int i = name.indexOf('%');
        while (i > -1 && i + 2 < name.length()) {
            buffer.append(name.toCharArray(), 0, i);
            int a = Character.digit(name.charAt(i + 1), 16);
            int b = Character.digit(name.charAt(i + 2), 16);
            if (a > -1 && b > -1) {
                buffer.append((char) (a * 16 + b));
                name = name.substring(i + 3);
            } else {
                buffer.append('%');
                name = name.substring(i + 1);
            }
            i = name.indexOf('%');
        }
        buffer.append(name);
        return buffer.toString();
    }
}
