package com.xebialabs.deployit.repository;

import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import org.joda.time.DateTime;

import javax.jcr.*;
import javax.jcr.nodetype.NodeType;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;
import java.util.Collection;

import static com.xebialabs.deployit.jcr.JcrConstants.*;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static javax.jcr.nodetype.NodeType.NT_FROZEN_NODE;
import static javax.jcr.query.Query.JCR_SQL2;

public class NodeUtils {

    static Node getReferencedCiNode(Node node, Value value, Session session) throws RepositoryException {
        String referencedCiUuid = value.getString();
        return getNodeByUuid(node, referencedCiUuid, session);
    }

    protected static Node getNodeByUuid(Node node, String referencedCiUuid, Session session) throws RepositoryException {
        try {
            return session.getNodeByIdentifier(referencedCiUuid);
        } catch (ItemNotFoundException exc) {
            if (nodeIsFrozen(node)) {
                Node frozenNode = getFrozenNodeByUuid(referencedCiUuid, session);
                if (frozenNode != null) {
                    return frozenNode;
                }
            }
            throw exc;
        }
    }

    static Node getFrozenNodeByUuid(String uuid, Session session) throws RepositoryException {
        QueryManager queryManager = session.getWorkspace().getQueryManager();
        ValueFactory valueFactory = session.getValueFactory();

        Query query = queryManager.createQuery("SELECT * FROM [" + NT_FROZEN_NODE + "] WHERE [" + Property.JCR_FROZEN_UUID + "] = $uuid", JCR_SQL2);
        query.bindValue("uuid", valueFactory.createValue(uuid));
        QueryResult result = query.execute();
        NodeIterator resultNodes = result.getNodes();
        if (resultNodes.hasNext()) {
            // Assuming that a query by UUID only return one node
            return resultNodes.nextNode();
        }
        return null;
    }

    static boolean nodeIsFrozen(Node node) throws RepositoryException {
        return node.isNodeType(NodeType.NT_FROZEN_NODE);
    }

    static Type typeOf(Node node) throws RepositoryException {
        return Type.valueOf(node.getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString());
    }

    static PropertyDescriptor descriptorOf(Property property) throws RepositoryException {
        return typeOf(property.getParent()).getDescriptor().getPropertyDescriptor(property.getName());
    }

    static PropertyDescriptor findContainmentListProperty(Type typeToScan, Type typeOfList) {
        Collection<PropertyDescriptor> propertyDescriptors = typeToScan.getDescriptor().getPropertyDescriptors();
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            if (propertyDescriptor.getKind() == LIST_OF_CI && propertyDescriptor.isAsContainment() && typeOfList.instanceOf(propertyDescriptor.getReferencedType())) {
                return propertyDescriptor;
            }
        }
        return null;
    }


    static CiAttributes readCiAttributes(Node node) throws RepositoryException {
        final DateTime createdAt = readDateProperty(node, CREATED_AT_PROPERTY_NAME);
        final DateTime lastModifiedAt = readDateProperty(node, LAST_MODIFIED_AT_PROPERTY_NAME);
        final String createdBy = readStringProperty(node, CREATED_BY_PROPERTY_NAME);
        final String lastModifiedBy = readStringProperty(node, LAST_MODIFIED_BY_PROPERTY_NAME);
        return new CiAttributes(
                (createdBy != null ? createdBy : lastModifiedBy),
                (createdAt != null ? createdAt : lastModifiedAt),
                lastModifiedBy,
                lastModifiedAt,
                null
        );
    }

    static String readStringProperty(Node node, String propertyName) throws RepositoryException {
        return node.hasProperty(propertyName) ? node.getProperty(propertyName).getString() : null;
    }

    static DateTime readDateProperty(Node node, String propertyName) throws RepositoryException {
        return node.hasProperty(propertyName) ? new DateTime(node.getProperty(propertyName).getDate().getTime()) : null;
    }
}
