package com.xebialabs.xlplatform.jcr

import java.util.concurrent.ExecutionException
import javax.jcr.{Repository, RepositoryException}

import com.xebialabs.xlplatform.repository.{RepositoryConfigurationSource, XlRepositoryConfig, XlRepositoryInitializer}
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import org.modeshape.jcr.{JcrRepository, ModeShapeEngine, RepositoryConfiguration}
import org.slf4j.{Logger, LoggerFactory}
import org.springframework.beans.factory.{DisposableBean, FactoryBean, InitializingBean}

class ModeShapeRepositoryFactoryBean(xlConfig: XlRepositoryConfig) extends InitializingBean with FactoryBean[Repository] with DisposableBean {

  private var engine: ModeShapeEngine = _

  private var repository: JcrRepository = _

  override def afterPropertiesSet(): Unit = {
    engine = new ModeShapeEngine
    engine.start()
    val modeShapeConfig: RepositoryConfiguration = readConfig(xlConfig.jcrRepositoryConfig)
    repository = engine.deploy(modeShapeConfig)
    new XlRepositoryInitializer(repository, xlConfig).init()
  }

  @throws(classOf[RepositoryException])
  private def readConfig(jcrConfigSource: RepositoryConfigurationSource): RepositoryConfiguration = {
    using(jcrConfigSource.getInputStream()) { stream =>
      try {
        val modeShapeConfig = RepositoryConfiguration.read(stream, jcrConfigSource.getName())
        val problems = modeShapeConfig.validate
        if (!problems.isEmpty) {
          import scala.collection.JavaConversions._
          for (problem <- problems) {
            logger.error(problem.getMessageString)
          }
          throw new RepositoryException("Could not boot due to problems")
        }
        modeShapeConfig
      }
      catch {
        case re: RepositoryException => throw re
        case e: Exception => throw new RepositoryException(e)
      }
    }
  }

  override def getObjectType: Class[_] = classOf[Repository]

  override def getObject: Repository = repository

  override def isSingleton: Boolean = true

  override def destroy(): Unit = {
    val shutdown = engine.shutdown()
    try {
      if (!shutdown.get) {
        throw new IllegalStateException("Did not shutdown...")
      }
    }
    catch {
      case e: InterruptedException =>
        Thread.currentThread.interrupt()
        throw new IllegalStateException("Failed to shutdown", e)
      case e: ExecutionException =>
        throw new IllegalStateException("Failed to shutdown", e)
    }
  }

  private val logger: Logger = LoggerFactory.getLogger(classOf[ModeShapeRepositoryFactoryBean])

}
