package com.xebialabs.xlplatform.jcr

import java.io.{File, FileInputStream, InputStream}
import java.util.concurrent.TimeUnit
import javax.jcr.RepositoryException

import com.typesafe.config.Config
import com.xebialabs.xlplatform.jcr.ModeshapeConfiguration.{AFile, AResource}
import com.xebialabs.xlplatform.utils.ClassLoaderUtils
import com.xebialabs.xlplatform.utils.ResourceManagement._
import grizzled.slf4j.Logging
import org.infinispan.configuration.cache.{CacheMode, Configuration, ConfigurationBuilder}
import org.infinispan.configuration.global.GlobalConfigurationBuilder
import org.infinispan.eviction.EvictionStrategy
import org.infinispan.manager.DefaultCacheManager
import org.infinispan.persistence.jdbc.configuration.JdbcStringBasedStoreConfigurationBuilder
import org.infinispan.schematic.document.Changes
import org.infinispan.schematic.internal.document.BasicDocument
import org.infinispan.transaction.lookup.{TransactionManagerLookup => InfinispanTransactionManagerLookup}
import org.infinispan.transaction.{LockingMode, TransactionMode}
import org.infinispan.util.concurrent.IsolationLevel
import org.modeshape.jcr.RepositoryConfiguration.FieldName
import org.modeshape.jcr.{LocalEnvironment, RepositoryConfiguration}


object ModeshapeConfiguration {
  def configHelper(repositoryCfg: Config) = new ConfigurationHelper(repositoryCfg)

  final val CONFIGURATION_LOCATION: String = "configurationLocation"

  private[jcr] class ConfigurationHelper(repository: Config) {

    lazy val configurationLocation: Option[String] = if (repository.hasPath(CONFIGURATION_LOCATION))
      Option(repository.getString(CONFIGURATION_LOCATION))
    else None

    object cluster {

      import collection.convert.wrapAsScala._

      private[this] lazy val cluster = repository.getConfig("cluster")

      lazy val name = cluster.getString("name")
      lazy val node = cluster.getString("node")
      lazy val enabled = cluster.getBoolean("enabled")
      lazy val hostname = cluster.getString("hostname")
      lazy val jGroupPort = cluster.getInt("port")
      lazy val members = cluster.getStringList("members").toList.mkString(",")
    }

    object persistence {
      private[this] lazy val persistence = repository.getConfig("persistence")

      lazy val driverName = persistence.getString("driverName")
      lazy val jdbcUrl = persistence.getString("jdbcUrl")
      lazy val user = persistence.getString("username")
      lazy val password = persistence.getString("password")
      lazy val columnTypes = driverName match {
        case d if d.toLowerCase.contains("oracle") => OracleColumnTypes
        case d if d.toLowerCase.contains("h2") => H2ColumnTypes
        case d if d.toLowerCase.contains("postgresql") => PostgresTypes
        case d if d.toLowerCase.contains("mysql") => MySqlTypes
      }
    }
  }

  trait ColumnTypes {
    val id: String
    val data: String
    val timestamp: String
  }

  object OracleColumnTypes extends ColumnTypes {
    override val id: String = "VARCHAR(255)"
    override val data: String = "BLOB"
    override val timestamp: String = "NUMBER(20)"
  }

  object H2ColumnTypes extends ColumnTypes {
    override val id: String = "VARCHAR(255)"
    override val data: String = "BINARY"
    override val timestamp: String = "BIGINT"
  }

  object PostgresTypes extends ColumnTypes {
    override val id: String = "VARCHAR(255)"
    override val data: String = "BYTEA"
    override val timestamp: String = "BIGINT"
  }

  object MySqlTypes extends ColumnTypes {
    override val id: String = "VARCHAR(255)"
    override val data: String = "BLOB"
    override val timestamp: String = "BIGINT"
  }

  object AFile {
    def unapply(path: String): Option[File] = {
      val file = new File(path)
      if (file.exists()) Option(file) else None
    }
  }

  object AResource {
    def unapply(resource: String): Option[InputStream] = {
      Option(ClassLoaderUtils.classLoader.getResourceAsStream(resource))
    }
  }

}

class ModeshapeConfiguration(repositoryName: String,
                             transactionManagerLookup: InfinispanTransactionManagerLookup,
                             config: Config) extends Logging {

  lazy val theConfig = ModeshapeConfiguration.configHelper(config)

  import theConfig._

  def modeShapeConfig: RepositoryConfiguration = {
    val configuration = configurationLocation.flatMap(loadConfiguration).getOrElse(programmaticConfiguration)
    logValidationAndThrowErrors(configuration)
    configuration
  }

  private def loadConfiguration(cfgLocation: String): Option[RepositoryConfiguration] = {
    loadConfigurationFile(cfgLocation).map(loadConfigurationFromStream(cfgLocation, _))
  }

  private def loadConfigurationFile(configuration: String): Option[InputStream] = configuration match {
    case AFile(file) => Some(new FileInputStream(file))
    case AResource(stream) => Some(stream)
    case _ => {
      logger.warn(s"Configuration location $configuration is not found on file system neither in the classpath")
      None
    }
  }

  private def loadConfigurationFromStream(configurationName: String, configurationInputStream: InputStream): RepositoryConfiguration = {
    using(configurationInputStream) { stream =>
      try {
        RepositoryConfiguration.read(stream, configurationName)
      }
      catch {
        case e: Exception => throw new RepositoryException(e)
      }
    }
  }

  private lazy val programmaticConfiguration = editModeshapeConfig(baseConfig).`with`(infinispanEnvironment)

  private lazy val baseConfig: RepositoryConfiguration = loadConfiguration("base.modeshape.config.json").get

  private def editModeshapeConfig(baseConfig: RepositoryConfiguration): RepositoryConfiguration = {
    val editor = baseConfig.withName(repositoryName).edit()
    editor.set(FieldName.STORAGE, cacheStorage)
    editor.set(FieldName.NAME, repositoryName)
    val config = applyChanges(baseConfig, editor.getChanges)
    config
  }

  private def cacheStorage = {
    new BasicDocument(FieldName.CACHE_NAME, repositoryName, FieldName.BINARY_STORAGE, binaryStorage)
  }

  private def binaryStorage = {
    new BasicDocument(FieldName.TYPE, "cache",
      FieldName.DATA_CACHE_NAME, repositoryName,
      FieldName.METADATA_CACHE_NAME, repositoryName,
      FieldName.MINIMUM_BINARY_SIZE_IN_BYTES, 4096)
  }

  private def applyChanges(repositoryConfiguration: RepositoryConfiguration, changes: Changes): RepositoryConfiguration = {
    val newChanges = repositoryConfiguration.edit()
    newChanges.apply(changes)
    new RepositoryConfiguration(newChanges.unwrap(), repositoryConfiguration.getName)
  }


  private def logValidationAndThrowErrors(config: RepositoryConfiguration): Unit = {
    import collection.convert.wrapAsScala._
    val problems = config.validate.toList
    if (problems.nonEmpty) {
      for (problem <- problems) {
        logger.error(problem.getMessageString)
      }
      throw new RepositoryException("Could not boot due to problems")
    }
  }

  private def infinispanEnvironment: LocalEnvironment = {
    val environment: LocalEnvironment = new LocalEnvironment()
    environment.addCacheContainer(LocalEnvironment.DEFAULT_CONFIGURATION_NAME, new DefaultCacheManager(globalInfinispanConfiguration))
    environment.defineCache(LocalEnvironment.DEFAULT_CONFIGURATION_NAME, repositoryName, infinispanConfig)
    environment
  }

  private lazy val globalInfinispanConfiguration = {
    val globalConfigurationBuilder: GlobalConfigurationBuilder = new GlobalConfigurationBuilder()
    if (cluster.enabled) {
      enableJGroupTransport(globalConfigurationBuilder)
    }
    globalConfigurationBuilder.globalJmxStatistics().enable().allowDuplicateDomains(true).cacheManagerName("XL-Infinispan")
    globalConfigurationBuilder.build()
  }

  private def enableJGroupTransport(globalConfigurationBuilder: GlobalConfigurationBuilder): Unit = {
    import cluster._
    System.setProperty("jgroups.tcp.address", hostname)
    System.setProperty("jgroups.tcp.port", jGroupPort.toString)
    System.setProperty("jgroups.tcp.cluster.addresses", members)
    globalConfigurationBuilder.transport().defaultTransport()
      .clusterName(name).nodeName(node)
      .addProperty("configurationFile", "tcpping-jgroups-tcp.xml")
  }

  private lazy val infinispanConfig: Configuration = {

    val configurationBuilder = new ConfigurationBuilder()

    if (cluster.enabled) {
      configurationBuilder.clustering().cacheMode(CacheMode.REPL_SYNC).sync()
        .stateTransfer().awaitInitialTransfer(true).timeout(60, TimeUnit.SECONDS)
    }

    configurationBuilder.eviction().strategy(EvictionStrategy.LRU).maxEntries(25000l)

    configurationBuilder.locking().isolationLevel(IsolationLevel.READ_COMMITTED)

    configurationBuilder.transaction()
      .transactionManagerLookup(transactionManagerLookup)
      .lockingMode(LockingMode.PESSIMISTIC)
      .transactionMode(TransactionMode.TRANSACTIONAL)

    configurePersistence(configurationBuilder)

    configurationBuilder.build(globalInfinispanConfiguration)
  }

  def configurePersistence(configurationBuilder: ConfigurationBuilder): Unit = {
    val jdbcStringBasedStoreConfigurationBuilder = new JdbcStringBasedStoreConfigurationBuilder(configurationBuilder.persistence())
    jdbcStringBasedStoreConfigurationBuilder.fetchPersistentState(true).purgeOnStartup(false).shared(true)
    jdbcStringBasedStoreConfigurationBuilder.connectionPool()
      .driverClass(persistence.driverName)
      .connectionUrl(persistence.jdbcUrl)
      .password(persistence.password)
      .username(persistence.user)
    jdbcStringBasedStoreConfigurationBuilder.table().dropOnExit(false).createOnStart(true).tableNamePrefix("ISPN_STRING")
      .idColumnName("ID").idColumnType(persistence.columnTypes.id)
      .dataColumnName("DATA").dataColumnType(persistence.columnTypes.data)
      .timestampColumnName("TIMESTAMP").timestampColumnType(persistence.columnTypes.timestamp)
    configurationBuilder.persistence().passivation(false).addStore(jdbcStringBasedStoreConfigurationBuilder)
  }
}
