package com.xebialabs.xlplatform.xlrepository.tck.suites

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot._
import com.xebialabs.deployit.repository.SearchParameters
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis._
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.Application
import scala.jdk.CollectionConverters._

trait RepositorySearchSuite extends RepositorySuiteBase { self: RepositoryFactory[_] =>
  import com.xebialabs.xlplatform.xlrepository.tck.util.CiHelper._

  private var ci: SampleCi = _
  private var otherCi: SampleCi = _
  private var subtypeCi: SampleSubclass = _
  private var parent: Parent = _

  describe("The XL Repository Search Suite") {

    it("should find one ConfigurationItem for specific type") {
      populateData()
      val searchParameters = new SearchParameters().setType(classOf[SampleSubclass])
      val ids = repository.list(searchParameters)
      ids should have size 1
    }

    it("should not find ConfigurationItems for type which is not in the repo") {
      populateData()
      val searchParameters = new SearchParameters().setType(classOf[Resource])
      val ids = repository.list(searchParameters)
      ids should have size 0
    }

    it("should find all ConfigurationItems and return them sorted alphabetically on the node name and not its full id") {
      populateData()
      val searchParameters: SearchParameters = new SearchParameters().setType(classOf[ConfigurationItem])
      val ids = repository.list(searchParameters).asScala.map(_.getId)

      ids should have size 12
      val oldCorrectOrder = Seq("Applications", "Configuration", "Environments", "Infrastructure", "Tests", "Configuration/a", "Configuration/a/b1", "Configuration/a/b2", "Configuration/a/b1/c", "Infrastructure/ci", "Infrastructure/otherCi", "Infrastructure/subtypeItem")

      ids should (equal(oldCorrectOrder) or equal(oldCorrectOrder.sorted))
    }

    it("should find three ConfigurationItems for type that also has subtypes") {
      populateData()
      val searchParameters = new SearchParameters().setType(classOf[SampleCi])
      val ids = repository.list(searchParameters)
      ids should have size 3
    }

    it("should find all ConfigurationItems for interface type") {
      populateData()
      val searchParameters = new SearchParameters().setType(classOf[ConfigurationItem])
      val ids = repository.list(searchParameters)
      ids should have size 12
    }

    describe("property search") {
      it("should find one ConfigurationItem with certain property") {
        populateData()
        val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("stringProperty", "aString")
        val ids = repository.list(params)
        ids should have size 1
        ids.get(0).getId should equal(ci.getId)
      }

      it("should find one ConfigurationItem using wildcard suffix") {
        populateData()
        val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("stringProperty", "a%")
        val ids = repository.list(params)
        ids should have size 1
        ids.get(0).getId should equal(ci.getId)
      }

      it("should find no ConfigurationItem using wrong wildcard search") {
        populateData()
        val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("stringProperty", "%tro%")
        val ids = repository.list(params)
        ids should have size 0
      }

      it("should find one ConfigurationItem using wildcard prefix") {
        populateData()
        val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("stringProperty", "%ing")
        val ids = repository.list(params)
        ids should have size 1
        ids.get(0).getId should equal(ci.getId)
      }

      it("should find one ConfigurationItem using wildcard around") {
        populateData()
        val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("stringProperty", "%a%tri%")
        val ids = repository.list(params)
        ids should have size 1
        ids.get(0).getId should equal(ci.getId)
      }

      it("should find one ConfigurationItem with certain property containing spaces") {
        ci = ciHelper.constructConfigurationItem(classOf[SampleCi], "Infrastructure/ci")
        ci.setStringProperty("Some value")
        repository.create(ci)

        val searchParameters: SearchParameters  = new SearchParameters().setType(classOf[SampleCi])
        searchParameters.addProperty("stringProperty", "Some value")
        val ids = repository.list(searchParameters)
        ids should have size 1
        ids.get(0).getId should equal(ci.getId)
      }
    }

    it("should find ConfigurationItem using wildcard on name") {
      populateData()
      val searchParameters = new SearchParameters().setName("%i")
      val ids = repository.list(searchParameters)
      ids should have size 2
    }

    it("should do case sensitive search on name without wildcard") {
      populateData()
      val searchParameters = new SearchParameters().setName(ci.getName)
      val ids = repository.list(searchParameters)
      ids should have size 1
    }

    it("should do case insensitive search on name wildcard") {
      populateData()
      val searchParameters = new SearchParameters().setName("%ci%")
      val ids = repository.list(searchParameters)
      ids should have size 2
    }

    it("should do case insensitive search on name wildcard with capitals in search") {
      populateData()
      val searchParameters = new SearchParameters().setName("%CI%")
      val ids = repository.list(searchParameters)
      ids should have size 2
    }

    it("should find root ConfigurationItems when setting parent to slash") {
      populateData()
      val ids = repository.list(new SearchParameters().setParent("/")).asScala.map(_.getId)
      ids should have size 5
      ids should contain(APPLICATIONS.getRootNodeName)
      ids should contain(ENVIRONMENTS.getRootNodeName)
      ids should contain(INFRASTRUCTURE.getRootNodeName)
      ids should contain(CONFIGURATION.getRootNodeName)
      ids should contain("Tests")
    }

    it("should find two ConfigurationItems when setting parent") {
      populateData()
      val searchParameters = new SearchParameters().setParent(parent.getId)
      val ids = repository.list(searchParameters)
      ids should have size 2
    }

    it("should find two ConfigurationItems when setting parent and type") {
      populateData()
      val searchParameters = new SearchParameters().setParent(parent.getId).setType(Type.valueOf(classOf[Child]))
      val ids = repository.list(searchParameters)
      ids should have size 2
    }

    it("should find 3 CIs when searching by ancestor") {
      populateData()
      val searchParameters = new SearchParameters().setAncestor(parent.getId)
      val ids = repository.list(searchParameters)
      ids should have size 3
    }

    it("should find 1 CI when searching by parent with pagination") {
      populateData()
      val searchParameters = new SearchParameters().setParent(parent.getId).setPage(1).setResultsPerPage(1)
      val ids = repository.list(searchParameters)
      ids should have size 1
      ids.get(0).getType shouldBe Type.valueOf(classOf[Child])
      ids.get(0).getId shouldBe parent.getId + "/b2"
    }

    it("should find ci using percent character in name") {
      populateDataForNameWithPercentCharacter()
      var otherCiName = otherCi.getName
      var searchParameters = new SearchParameters().setName("c\\%i")
      var ids = repository.list(searchParameters)
      ids should have size 1

      searchParameters = new SearchParameters().setName("otherC\\%i")
      ids = repository.list(searchParameters)
      ids should have size 1
    }

    it("should do case sensitive search on name with percent character") {
      populateDataForNameWithPercentCharacter()
      var ciName = ci.getName.replace("%", "\\%")
      var searchParameters = new SearchParameters().setName(ciName)
      var ids = repository.list(searchParameters)
      ids should have size 1

      searchParameters = new SearchParameters().setName(ciName.toUpperCase())
      ids = repository.list(searchParameters)
      ids should have size 0

    }

    it("should do case sensitive search on name without percent character for application ci") {
      populateDataForNameWithPercentCharacter()
      var searchParameters = new SearchParameters().setName("AppForTest")
      var ids = repository.list(searchParameters)
      ids should have size 1

      searchParameters = new SearchParameters().setName("App\\%ForTest")
      ids = repository.list(searchParameters)
      ids should have size 0

      searchParameters = new SearchParameters().setName("App\\%\\%\\%ForTest")
      ids = repository.list(searchParameters)
      ids should have size 0
    }

    it("should allow creation of application with percent character") {
      repository.create(ciHelper.constructConfigurationItem(classOf[Application], "Applications/AppForTest"))
      try {
        repository.create(ciHelper.constructConfigurationItem(classOf[Application], "Applications/AppForTest"))
        fail("Should not be able to create application with existing name: AppForTest")
      } catch {
        case t: Throwable =>
          succeed
      }

      repository.create(ciHelper.constructConfigurationItem(classOf[Application], "Applications/App%ForTest"))
      try {
        repository.create(ciHelper.constructConfigurationItem(classOf[Application], "Applications/App%ForTest"))
        fail("Should not be able to create application with existing name: AppForTest")
      } catch {
        case t: Throwable =>
          succeed
      }

      repository.create(ciHelper.constructConfigurationItem(classOf[Application], "Applications/App%%ForTest"))
      repository.create(ciHelper.constructConfigurationItem(classOf[Application], "Applications/app%fortest"))
      succeed
    }

  }

  private def populateData(): Unit = {
    ci = ciHelper.constructConfigurationItem(classOf[SampleCi], "Infrastructure/ci")
    ci.setStringProperty("aString")
    repository.create(ci)

    otherCi = ciHelper.constructConfigurationItem(classOf[SampleCi], "Infrastructure/otherCi")
    otherCi.setIntProperty(123)
    repository.create(otherCi)

    subtypeCi = ciHelper.constructConfigurationItem(classOf[SampleSubclass], "Infrastructure/subtypeItem")
    subtypeCi.setStringProperty("stringInTheSubtype")
    repository.create(subtypeCi)

    parent = ciHelper.constructConfigurationItem(classOf[Parent], "Configuration/a")
    repository.create(parent)

    val b1 = ciHelper.constructConfigurationItem(classOf[Child], parent.getId + "/b1")
    b1.prop = "test"
    repository.create(b1)

    val b2 = ciHelper.constructConfigurationItem(classOf[Child], parent.getId + "/b2")
    repository.create(b2)

    val c = ciHelper.constructConfigurationItem(classOf[GrandChild], b1.getId + "/c")
    repository.create(c)
  }

  private def populateDataForNameWithPercentCharacter(): Unit = {
    ci = ciHelper.constructConfigurationItem(classOf[SampleCi], "Infrastructure/c%i")
    ci.setStringProperty("aString")
    repository.create(ci)

    otherCi = ciHelper.constructConfigurationItem(classOf[SampleCi], "Infrastructure/otherC%i")
    otherCi.setIntProperty(123)
    repository.create(otherCi)

    subtypeCi = ciHelper.constructConfigurationItem(classOf[SampleSubclass], "Infrastructure/subtypeItem")
    subtypeCi.setStringProperty("stringInTheSubtype")
    repository.create(subtypeCi)

    parent = ciHelper.constructConfigurationItem(classOf[Parent], "Configuration/a")
    repository.create(parent)

    val b1 = ciHelper.constructConfigurationItem(classOf[Child], parent.getId + "/b1")
    b1.prop = "test"
    repository.create(b1)

    val b2 = ciHelper.constructConfigurationItem(classOf[Child], parent.getId + "/b2")
    repository.create(b2)

    val c = ciHelper.constructConfigurationItem(classOf[GrandChild], b1.getId + "/c")
    repository.create(c)

    repository.create(ciHelper.constructConfigurationItem(classOf[Application], "Applications/AppForTest"))

  }
}
