package com.xebialabs.xlplatform.xlrepository.tck.suites

import java.io.{ByteArrayOutputStream, File}
import java.lang.String._
import java.net.{URL, URISyntaxException}
import java.util.zip.{ZipEntry, ZipOutputStream}

import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.deployit.io.DerivedArtifactFile
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.repository.WorkDir
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.{OverthereUtils, ByteArrayFile}
import com.xebialabs.xlplatform.test.{file, folder}
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis.{SampleContainer, SampleDerivedFile, SampleFile, SampleFolder}
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase.RichCi

import scala.collection.convert.wrapAll._
import scala.io.Source

trait ArtifactSuite extends RepositorySuiteBase {
  self: RepositoryFactory[_] =>

  private val gen = generator.genFor("tck-artifact")

  describe("The Artifact suite") {
    it("should read / write a File artifact") {
      val artifactData: Array[Byte] = "**********".getBytes
      val artifact: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), new ByteArrayFile("file-1.0.ext", artifactData))
      repository.create(artifact)
      val workDir: WorkDir = workDirFactory.newWorkDir
      val read: SampleFile = repository.read(artifact.getId, workDir)
      val file: OverthereFile = read.getFile
      file shouldNot be(null)
      file.getPath should startWith(workDir.getPath)
      file.getName shouldEqual "file-1.0.ext"
      using(file.getInputStream) { is =>
        Source.fromInputStream(is).map(_.toByte).toArray shouldEqual artifactData
      }
    }

    ignore("should create new file if artifact is read second time in different context") {
      // Why does this fail on the console, while it works in IDEA?
      val artifactData: Array[Byte] = "**********".getBytes
      val artifact: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), new ByteArrayFile("file-2.0.ext", artifactData))
      repository.create(artifact)

      val workDir: WorkDir = workDirFactory.newWorkDir
      val read: SampleFile = repository.read(artifact.getId, workDir)
      val workDir2: WorkDir = workDirFactory.newWorkDir
      val read2: SampleFile = repository.read(artifact.getId, workDir2)
      workDir.getPath shouldNot equal(workDir2.getPath)

      read.getFile shouldNot be(theSameInstanceAs(read2.getFile))
      read.getFile.getPath shouldNot equal(read2.getFile.getPath)
      read.getFile.getName shouldEqual read2.getFile.getName
    }

    it("should read / write a Folder artifact") {
      val artifactFolder: File = mk(folder("config", file("config1.properties", "key=value\nanotherkey=anothervalue\n")))
      val artifact: SampleFolder = ciHelper.constructArtifact(classOf[SampleFolder], gen.nextId(), LocalFile.valueOf(artifactFolder))
      repository.create(artifact)
      val workDir: WorkDir = workDirFactory.newWorkDir
      val read: SampleFolder = repository.read(artifact.getId, workDir)
      read.getFile shouldBe 'directory
      read.getFile.getName shouldEqual "config"
    }

    it("should deal with derived artifacts which do not have a source artifact counterpart set") {
      val container: SampleContainer = ciHelper.constructConfigurationItem(classOf[SampleContainer], gen.nextId())
      val artifact: SampleDerivedFile = ciHelper.constructArtifact(classOf[SampleDerivedFile], gen.subId(container), null)
      repository.create(artifact, container)
      val read: SampleDerivedFile = repository.read(artifact.getId, workDirFactory.newWorkDir)
      read.getFile shouldBe a[DerivedArtifactFile]
    }

    it("should not read the file for a derived artifact when there is no workdir set") {
      val source: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), new ByteArrayFile("test.txt", "**********".getBytes()))
      val container: SampleContainer = ciHelper.constructConfigurationItem(classOf[SampleContainer], gen.nextId())
      val artifact: SampleDerivedFile = ciHelper.constructArtifact(classOf[SampleDerivedFile], gen.subId(container), null)
      artifact.setDeployable(source)
      repository.create(source, container, artifact)
      val read: SampleDerivedFile = repository.read(artifact.getId)
      read.getFile shouldBe a[DerivedArtifactFile]
    }

    it("should create a derived artifact that replaces the contents with the provided placeholders") {
      val source: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), new ByteArrayFile("foobarbaz.txt", "{{foo}}bar{{baz}}\n".getBytes()))
      val container: SampleContainer = ciHelper.constructConfigurationItem(classOf[SampleContainer], gen.nextId())
      val artifact: SampleDerivedFile = ciHelper.constructArtifact(classOf[SampleDerivedFile], gen.subId(container), null)
      artifact.setDeployable(source)
      artifact.setPlaceholders(Map("foo" -> "r1 ", "baz" -> " r2"))
      repository.create(source, container, artifact)
      val read: SampleDerivedFile = repository.read(artifact.getId, workDirFactory.newWorkDir)
      using(read.getFile.getInputStream) { is =>
        Source.fromInputStream(is).map(_.toByte).toArray shouldEqual "r1 bar r2\n".getBytes()
      }
    }

    it("should calculate the checksum on create if it is not set") {
      val artifact: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), new ByteArrayFile("checksum.txt", "**********".getBytes()))
      artifact.getChecksum shouldEqual null
      repository.create(artifact)
      artifact.getChecksum should not(equal(null)).and(not(equal("")))
    }

    it("should not create artifact without data") {
      val artifact: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), null)
      an[IncorrectArgumentException] shouldBe thrownBy {
        repository.create(artifact)
      }
    }

    it("should create artifact which references an external artifact") {
      val artifact: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), new ByteArrayFile("original.txt", "**********".getBytes()))
      repository.create(artifact)
      val externalArtifact: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), null)
      externalArtifact.setProperty(SourceArtifact.FILE_URI_PROPERTY_NAME, s"jcrref:${artifact.getId}/original.txt")
      // Normally this is handled by the ArtifactEnricher
      externalArtifact.setFile(artifact.getFile)
      repository.create(externalArtifact)
    }

    it("should let create an artifact by copying an existing one") {
      val content: Array[Byte] = "**********".getBytes()
      val artifact: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), new ByteArrayFile("original.txt", content))
      repository.create(artifact)

      val artifactToCopy: SampleFile = repository.read(artifact.getId, workDirFactory.newWorkDir())
      artifactToCopy.getFile.getPath // makes the SourceArtifact resolved: resolution is the responsibility of user of repository service
      artifactToCopy.setId(gen.nextId())
      artifactToCopy.set$token(null)
      repository.create(artifactToCopy)

      val copiedArtifact: SampleFile = repository.read(artifactToCopy.getId, workDirFactory.newWorkDir())
      using(copiedArtifact.getFile.getInputStream) { is =>
        Source.fromInputStream(is).map(_.toByte).toArray shouldEqual content
      }
    }

    it("should update an artifact without passing new data") {
      val artifact: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], gen.nextId(), new ByteArrayFile("original.txt", "**********".getBytes()))
      repository.create(artifact)
      val newArtifact: SampleFile = ciHelper.constructArtifact(classOf[SampleFile], artifact.getId, null)
      newArtifact.string = "new value"
      newArtifact.getFile shouldEqual null
      repository.update(newArtifact)
      artifact property "string" shouldBe (storedInRepository equal "new value")
      artifact property "fileUri" shouldBe (storedInRepository equal artifact.getFileUri)
    }

    it("should calculate same checksum for folder as zip that only differs in zip name") {
      val file1: OverthereFile = zip(mk(folder("initial", folder("texts", file("test.txt", "This is a test.")))))
      val file2: OverthereFile = zip(mk(folder("same", folder("texts", file("test.txt", "This is a test.")))))
      shouldCalculateEqualChecksum(file1, file2)
    }

    it("should calculate checksum correctly on for different zipped folder") {
      val file1 = zip(mk(folder("initial", folder("texts", file("test.txt", "This is a test.")))))
      val file2 = zip(mk(folder("differ", folder("texts", file("test.txt", "This is a test. With a modification.")))))
      shouldCalculateDifferentChecksum(file1, file2)
    }

    it("should calculate same checksum for complex folder as zip that only differs in zip name") {
      val file1 = zip(mk(folder("initial-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir"))))
      val file2 = zip(mk(folder("same-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir"))))
      shouldCalculateEqualChecksum(file1, file2)
    }

    it("should calculate different checksum for complex folder as zip that has moved a file") {
      val file1 = zip(mk(folder("initial-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir"))))
      val file2 = zip(mk(folder("differ-complex", folder("texts2", file("test.txt", "This is a test.")), folder("empty-dir"))))
      shouldCalculateDifferentChecksum(file1, file2)
    }

    it("should calculate different checksum for complex folder as zip that has a new empty file") {
      val file1 = zip(mk(folder("initial-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir"))))
      val file2 = zip(mk(folder("differ-complex", folder("texts2", file("test.txt", "This is a test.")), folder("empty-dir"), file("empty-file"))))
      shouldCalculateDifferentChecksum(file1, file2)
    }

    it("should calculate different checksum for complex folder as zip that has a missing empty dir") {
      val file1 = zip(mk(folder("initial-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir"))))
      val file2 = zip(mk(folder("differ-complex", folder("texts2", file("test.txt", "This is a test.")))))
      shouldCalculateDifferentChecksum(file1, file2)
    }

    it("should calculate same checksum for folder that only differs in folder name") {
      val file1: File = mk(folder("initial", folder("texts", file("test.txt", "This is a test."))))
      val file2: File = mk(folder("same", folder("texts", file("test.txt", "This is a test."))))
      shouldCalculateEqualChecksum(LocalFile.valueOf(file1), LocalFile.valueOf(file2))
    }

    it("should calculate checksum correctly for different folder") {
      val file1: File = mk(folder("initial", folder("texts", file("test.txt", "This is a test."))))
      val file2: File = mk(folder("differ", folder("texts", file("test.txt", "This is a test. With a modification."))))
      shouldCalculateDifferentChecksum(LocalFile.valueOf(file1), LocalFile.valueOf(file2))
    }

    it("should calculate same checksum for complex folder that only differs in folder name") {
      val file1: File = mk(folder("initial-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir")))
      val file2: File = mk(folder("same-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir")))
      shouldCalculateEqualChecksum(LocalFile.valueOf(file1), LocalFile.valueOf(file2))
    }

    it("should calculate different checksum for complex folder that has moved a file") {
      val file1: File = mk(folder("initial-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir")))
      val file2: File = mk(folder("differ-complex", folder("texts2", file("test.txt", "This is a test.")), folder("empty-dir")))
      shouldCalculateDifferentChecksum(LocalFile.valueOf(file1), LocalFile.valueOf(file2))
    }

    it("should calculate different checksum for complex folder that has a new empty file") {
      val file1: File = mk(folder("initial-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir")))
      val file2: File = mk(folder("differ-complex", folder("texts2", file("test.txt", "This is a test.")), folder("empty-dir"), file("empty-file")))
      shouldCalculateDifferentChecksum(LocalFile.valueOf(file1), LocalFile.valueOf(file2))
    }

    it("should calculate different checksum for complex folder that has a missing empty dir") {
      val file1: File = mk(folder("initial-complex", folder("texts", file("test.txt", "This is a test.")), folder("empty-dir")))
      val file2: File = mk(folder("differ-complex", folder("texts2", file("test.txt", "This is a test."))))
      shouldCalculateDifferentChecksum(LocalFile.valueOf(file1), LocalFile.valueOf(file2))
    }

    it("should store empty folder artifact") {
      val file = mk(folder("empty"))
      val artifact: SampleFolder = ciHelper.createArtifact(classOf[SampleFolder], gen.nextId(), LocalFile.valueOf(file))
      val read: SampleFolder = repository.read(artifact.getId, workDirFactory.newWorkDir())
      read.getFile shouldNot equal(null)
      read.getFile.listFiles() shouldBe empty
    }
  }

  private[this] def shouldCalculateDifferentChecksum(file1: OverthereFile, file2: OverthereFile): Unit = {
    val artifact: SampleFolder = ciHelper.createArtifact(classOf[SampleFolder], gen.nextId(), file1)
    val same: SampleFolder = ciHelper.createArtifact(classOf[SampleFolder], gen.nextId(), file2)
    artifact.getChecksum should not(equal(null).and(not(equal(""))))
    artifact.getChecksum should not(equal(same.getChecksum))
  }

  private[this] def shouldCalculateEqualChecksum(file1: OverthereFile, file2: OverthereFile): Unit = {
    val artifact: SampleFolder = ciHelper.createArtifact(classOf[SampleFolder], gen.nextId(), file1)
    val same: SampleFolder = ciHelper.createArtifact(classOf[SampleFolder], gen.nextId(), file2)
    artifact.getChecksum should not(equal(null).and(not(equal(""))))
    artifact.getChecksum shouldEqual same.getChecksum
  }

  private[this] def getResource(resourcePath: String): OverthereFile = {
    try {
      val resource: URL = Thread.currentThread.getContextClassLoader.getResource(resourcePath)
      using(resource.openStream()) { is =>
        val baos = new ByteArrayOutputStream()
        OverthereUtils.write(is, baos)
        new ByteArrayFile(resourcePath, baos.toByteArray)
      }
    }
    catch {
      case e: URISyntaxException => {
        throw new RuntimeException(format("Invalid resource path: %s", resourcePath), e)
      }
    }
  }

  private[this] def zip(f: File): OverthereFile = {
    val localFile: OverthereFile = LocalFile.valueOf(f)
    val baos: ByteArrayOutputStream = new ByteArrayOutputStream()
    val os: ZipOutputStream = new ZipOutputStream(baos)
    def writeEntry(f: OverthereFile, path: String, stream: ZipOutputStream): Unit = {
      if (f.isFile) {
        stream.putNextEntry(new ZipEntry(path + f.getName))
        using(f.getInputStream) { is =>
          OverthereUtils.write(is, stream)
        }
        stream.closeEntry()
      } else if (f.isDirectory) {
        stream.putNextEntry(new ZipEntry(path + f.getName + "/"))
        stream.closeEntry()
        f.listFiles().foreach { subFile =>
          writeEntry(subFile, path + f.getName + "/", stream)
        }
      }
    }

    localFile.listFiles().foreach { f =>
      writeEntry(f, "", os)
    }
    new ByteArrayFile(f.getPath + ".zip", baos.toByteArray)
  }
}
