package com.xebialabs.xlplatform.xlrepository.tck.suites

import com.google.common.collect.Lists._
import com.google.common.collect.Sets._
import com.google.common.io.ByteSource
import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.udm.artifact.{Artifact, SourceArtifact}
import com.xebialabs.deployit.repository._
import com.xebialabs.xlplatform.utils.ResourceUtils.asByteSource
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis._
import com.xebialabs.xlplatform.xlrepository.tck.util.{CiHelper, RepositorySuiteBase}

import scala.collection.convert.wrapAll._

trait RepositoryCrudSuite extends RepositorySuiteBase { self: RepositoryFactory[_] =>

  private val gen = generator.genFor("tck-crud")

  import CiHelper._
  describe("The XL Repository Crud Suite") {

    it("should not create ConfigurationItem with incorrect reference") {
      val ci = ciHelper.constructConfigurationItem(classOf[Referencer], gen.nextId())
      ci.ref = ciHelper.constructConfigurationItem(classOf[Referencer], gen.nextId())

      an[Exception] should be thrownBy {
        repository.create(ci)
      }
    }

    it("should persist ConfigurationItems with references in the repository") {
      val referredConfigurationItem = ciHelper.createConfigurationItem(classOf[MultiReferencer], gen.nextId())
      val anotherReferredConfigurationItem = ciHelper.createConfigurationItem(classOf[MultiReferencer], gen.nextId())

      val createdConfigurationItem = createConfigurationItemWithRerefence(referredConfigurationItem, anotherReferredConfigurationItem)
      assertInRepository(3, createdConfigurationItem)

      val err = intercept[ItemInUseException] {
        repository.delete(referredConfigurationItem.getId)
      }
      err.getMessage should include(referredConfigurationItem.getId)
      repository.delete(createdConfigurationItem.getId, anotherReferredConfigurationItem.getId, referredConfigurationItem.getId)
    }

    it("should get ordered ConfigurationItem list") {
      val anotherConfigurationItem = ciHelper.constructConfigurationItem(classOf[Basic], gen.nextId())
      val configurationItem = ciHelper.constructConfigurationItem(classOf[Basic], gen.nextId())
      repository.create(configurationItem, anotherConfigurationItem)

      val createdOrderedEntities = repository.list(new SearchParameters)
      createdOrderedEntities.size shouldBe 7
      val firstId = createdOrderedEntities.get(5)
      firstId.getId should equal (anotherConfigurationItem.getId)
      val secondId = createdOrderedEntities.get(6)
      secondId.getId should equal (configurationItem.getId)

      repository.delete(anotherConfigurationItem.getId)
      repository.delete(configurationItem.getId)
    }

    it("should find descendants") {
      val tree1 = createCiTree("shouldFindDescendant1")
      createCiTree("shouldFindDescendant2")
      val pkgs = repository.list(new SearchParameters().setAncestor(tree1("app")).setType(Type.valueOf(classOf[Resource])))
      pkgs should have size 1
    }

    it("should throw exception when storing child ConfigurationItem under incorrect node") {
      val incorrectParent = ciHelper.constructConfigurationItem(classOf[Rooted],
        "Infrastructure/shouldThrowExceptionWhenStoringChildCiUnderIncorrectNode")
      repository.create(incorrectParent)

      val dummyInfrastructureChild = ciHelper.constructConfigurationItem(classOf[Child], incorrectParent.getId + "/dummy-infra-child")
      an[IncorrectArgumentException] should be thrownBy {
        repository.create(dummyInfrastructureChild)
      }
    }

    it("should not create the same ConfigurationItem twice") {
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", gen.nextId())
      repository.create(ci)
      an[ItemAlreadyExistsException] should be thrownBy {
        repository.create(ci)
      }
    }

    it("should persist a ConfigurationItem in the repository") {
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", gen.nextId())
      ci shouldNot existInRepository
      repository.create(ci)
      ci should existInRepository
      repository.update(ci)
      ci should existInRepository
      repository.delete(ci.getId)
      ci shouldNot existInRepository
    }

    it("should not fail on deleting non-existing ConfigurationItem") {
      repository.delete(gen.nextId())
    }

    it("should throw an exception when trying to read a non-existing ConfigurationItem") {
      an[NotFoundException] should be thrownBy {
        repository.read(gen.nextId())
      }
    }

    it("should recursively delete a ConfigurationItem tree") {
      val parent: ConfigurationItem = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      repository.create(parent)
      val child: ConfigurationItem = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
      repository.create(child)
      repository.delete(parent.getId)
      child shouldNot existInRepository
    }

    it("should move a ConfigurationItem to a new name") {
      val oldId: String = gen.nextId()
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", oldId)
      repository.create(ci)
      val newId: String = gen.nextId()
      repository.move(oldId, newId)
      oldId shouldNot existInRepository
      newId should existInRepository
    }

    it("should not move a ConfigurationItem to an existing id") {
      val oldId: String = gen.nextId()
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", oldId)
      val ci2: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", gen.nextId())
      repository.create(ci, ci2)
      val newId: String = ci2.getId
      an[ItemAlreadyExistsException] should be thrownBy {
        repository.move(oldId, newId)
      }
      oldId should existInRepository
      newId should existInRepository
    }

    it("should not move a ConfigurationItem into a wrong tree") {
      val oldId: String = gen.nextId()
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", oldId)
      repository.create(ci)
      val newId: String = "Environments/not-here"
      an[IncorrectArgumentException] should be thrownBy {
        repository.move(oldId, newId)
      }
      oldId should existInRepository
      newId shouldNot existInRepository
    }

    it("should not move a ConfigurationItem to same location") {
      val oldId: String = gen.nextId()
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", oldId)
      repository.create(ci)
      an[IncorrectArgumentException] should be thrownBy {
        repository.move(oldId, oldId)
      }
      oldId should existInRepository
    }

    it("should be able to move a ConfigurationItem into another (valid) sub-tree") {
      val parent: ConfigurationItem = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      repository.create(parent)
      val parent2: ConfigurationItem = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      repository.create(parent2)
      val child: ConfigurationItem = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
      repository.create(child)
      val id: String = gen.subId(parent2)
      repository.move(child.getId, id)
      child.getId shouldNot existInRepository
      id should existInRepository
    }

    it("should be move a child ConfigurationItems together with their moved parent") {
      val parent: ConfigurationItem = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      repository.create(parent)
      val child: ConfigurationItem = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
      repository.create(child)
      val id: String = gen.nextId()
      repository.move(parent.getId, id)
      parent.getId shouldNot existInRepository
      child.getId shouldNot existInRepository
      id should existInRepository
      s"$id/${child.getName}" should existInRepository
    }

    it("should be able to rename a rooted ConfigurationItem") {
      val oldId: String = gen.nextId()
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", oldId)
      repository.create(ci)
      repository.rename(oldId, "new-name")
      ci shouldNot existInRepository
      "Configuration/new-name" should existInRepository
    }

    it("should be able to rename a non-rooted ConfigurationItem") {
      val parent: ConfigurationItem = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      repository.create(parent)
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
      repository.create(ci)
      repository.rename(ci.getId, "new-name")
      ci shouldNot existInRepository
      s"${parent.getId}/new-name" should existInRepository
    }

    it("should not be possible to rename a ConfigurationItem into another tree") {
      val oldId: String = gen.nextId()
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", oldId)
      repository.create(ci)
      an[IncorrectArgumentException] should be thrownBy {
        repository.rename(oldId, gen.nextId())
      }
      ci should existInRepository
    }

    it("should not be possible to rename a ConfigurationItem to an existing name") {
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", gen.nextId())
      val ci2: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", gen.nextId())
      repository.create(ci, ci2)
      an[ItemAlreadyExistsException] should be thrownBy {
        repository.rename(ci.getId, ci2.getName)
      }
      ci should existInRepository
    }

    it("should correctly read a CI with depth 0") {
      val parent: Parent = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      parent.prop = "Foo"
      repository.create(parent)
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
      repository.create(ci)

      val read: Parent = repository.read(parent.getId, 0)
      read.prop shouldEqual null
      read.myChildren shouldBe 'empty
      read.getId shouldEqual parent.getId
      read.getType shouldEqual parent.getType
      read.get$ciAttributes().getCreatedAt shouldEqual null
      read.get$ciAttributes().getCreatedBy shouldEqual null
      read.get$ciAttributes().getLastModifiedAt shouldEqual null
      read.get$ciAttributes().getLastModifiedBy shouldEqual null
    }

    it("should correctly read a CI with depth 1") {
      val parent: Parent = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      parent.prop = "Foo"
      repository.create(parent)
      val ci: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
      ci.prop = "Bar"
      repository.create(ci)

      val read: Parent = repository.read(parent.getId, 1)
      read.prop shouldEqual "Foo"
      read.myChildren should have size 1
      read.myChildren.head.prop shouldEqual null
      read.myChildren.head.getId shouldEqual ci.getId
      read.get$ciAttributes().getCreatedAt should not equal null
      read.get$ciAttributes().getCreatedBy should not equal null
      read.get$ciAttributes().getLastModifiedAt should not equal null
      read.get$ciAttributes().getLastModifiedBy should not equal null
    }

    it("should correctly read a CI with depth 2") {
      val parent: Parent = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      parent.prop = "Foo"
      repository.create(parent)
      val ci: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
      ci.prop = "Bar"
      repository.create(ci)

      val read: Parent = repository.read(parent.getId, 2)
      read.prop shouldEqual "Foo"
      read.myChildren should have size 1
      read.myChildren.head.prop shouldEqual "Bar"
    }
  }

  private def createConfigurationItemWithRerefence(referedConfigurationItem: MultiReferencer,
                                                   anotherReferedConfigurationItem: MultiReferencer): MultiReferencer = {
    val ci = ciHelper.constructConfigurationItem(classOf[MultiReferencer], gen.nextId())
    ci.ref = referedConfigurationItem
    ci.refSet = newHashSet(referedConfigurationItem, anotherReferedConfigurationItem)
    repository.create(ci)
    ci
  }

  protected def assertInRepository[T <: ConfigurationItem](expectedNumberOfEntities: Int, expectedEntity: T) {
    val searchParameters = new SearchParameters().setType(expectedEntity.getType)
    val entities = repository.list(searchParameters)
    expectedNumberOfEntities should equal(entities.size)
    entities should contain(new ConfigurationItemData(expectedEntity.getId, expectedEntity.getType))
    val actualEntity: T = repository.read[T](expectedEntity.getId, workDirFactory.newWorkDir())
    assertEntityEquals(expectedEntity, actualEntity)
  }

  private def assertEntityEquals[T <: ConfigurationItem](expectedEntity: T, actualEntity: T) {
    expectedEntity.getType should equal(actualEntity.getType)
    assertPropertiesEqual(expectedEntity, actualEntity)
    expectedEntity.getClass should equal(actualEntity.getClass)
    expectedEntity match {
      case expectedArtifact: Artifact =>
        val actualArtifact = actualEntity.asInstanceOf[Artifact]
        if (expectedArtifact.getFile == null && expectedArtifact.isInstanceOf[SourceArtifact]) {
          val expectedSa = expectedArtifact.asInstanceOf[SourceArtifact]
          val actualSa = actualArtifact.asInstanceOf[SourceArtifact]
          expectedSa.getFileUri should equal(actualSa.getFileUri)
        }
        else {
          expectedArtifact.getFile shouldBe null
          actualArtifact.getFile should not be null
          actualArtifact.getFile.getName should equal(expectedArtifact.getFile.getName)
          val expectedData = getByteSource(expectedArtifact).read
          val actualData = getByteSource(actualArtifact).read
          expectedData should equal(actualData)
        }
      case _ =>
    }
  }

  private def assertPropertiesEqual[T <: ConfigurationItem](expectedEntity: T, actualEntity: T) {
    val descriptor = DescriptorRegistry.getDescriptor(expectedEntity.getType)
    descriptor.getPropertyDescriptors.foreach { pd =>
      pd.get(expectedEntity) should equal(pd.get(actualEntity))
    }
  }

  private def getByteSource(expectedArtifact: Artifact): ByteSource = {
    asByteSource(expectedArtifact.getFile)
  }

  private def createCiTree(testName: String): Map[String, String] = {
    val app = ciHelper.createConfigurationItem(classOf[Application], "Applications/app-" + testName)
    val pkg = ciHelper.createConfigurationItem(classOf[DeploymentPackage], app.getId + "/pkg-" + testName)
    ciHelper.createConfigurationItem(classOf[Resource], pkg.getId + "/resource" + testName)
    val cont = ciHelper.createConfigurationItem(classOf[SampleContainer], "Configuration/cont-" + testName)
    val dict = ciHelper.createConfigurationItem(classOf[Dictionary], "Environments/dict-" + testName)
    val env = ciHelper.createConfigurationItem(classOf[Environment], "Environments/env-" + testName)
    env.setMembers(newHashSet(cont))
    env.setDictionaries(newArrayList(dict))
    repository.update(env)
    val depApp = ciHelper.createConfigurationItem(classOf[DeployedApplication], env.getId + "/" + app.getName)
    depApp.setVersion(pkg)
    depApp.setOrchestrator(newArrayList("dummy-orchestrator"))
    repository.update(depApp)
    Map("app" -> app.getId, "pkg" -> pkg.getId, "env" -> env.getId, "depApp" -> depApp.getId)
  }
}
