package com.xebialabs.xlplatform.xlrepository.tck.suites

import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.deployit.engine.spi.event._
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis.{SampleCi, SampleContainer}
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase
import nl.javadude.t2bus.Subscribe


trait RepositoryEventSuite extends RepositorySuiteBase { self: RepositoryFactory[_] =>

  private val gen = generator.genFor("event")
  private var listener: EventCountingListener = _
  describe("The XL Event suite") {

    it("should fire event for ConfigurationItem creation and deletion") {
      subscribe()
      val ci = ciHelper.constructConfigurationItem(classOf[SampleContainer], gen.nextId())
      repository.create(ci)
      listener.totalCount shouldBe 1
      listener.countPerEventClass(classOf[CisCreatedEvent]) shouldBe 1
      listener.eventsPerEventClass(classOf[CisCreatedEvent]).head
        .asInstanceOf[CisCreatedEvent].getCis.get(0) should equal (ci)

      repository.delete(ci.getId)
      listener.totalCount shouldBe 2
      listener.countPerEventClass(classOf[CisDeletedEvent]) shouldBe 1
      listener.eventsPerEventClass(classOf[CisDeletedEvent]).head
        .asInstanceOf[CisDeletedEvent].getIds.head should equal (ci.getId)
      unsubscribe()
    }

    it("should fire event with multiple ConfigurationItems") {
      subscribe()
      val ci1 = ciHelper.constructConfigurationItem(classOf[SampleContainer], gen.nextId())
      val ci2 = ciHelper.constructConfigurationItem(classOf[SampleContainer], gen.nextId())
      repository.create(ci1, ci2)
      listener.totalCount shouldBe 1
      listener.countPerEventClass(classOf[CisCreatedEvent]) shouldBe 1
      listener.eventsPerEventClass(classOf[CisCreatedEvent]).head.asInstanceOf[CisCreatedEvent].getCis.get(0) should equal (ci1)
      listener.eventsPerEventClass(classOf[CisCreatedEvent]).head.asInstanceOf[CisCreatedEvent].getCis.get(1) should equal (ci2)

      repository.delete(ci1.getId, ci2.getId)
      listener.totalCount shouldBe 2
      listener.countPerEventClass(classOf[CisDeletedEvent]) shouldBe 1
      listener.eventsPerEventClass(classOf[CisDeletedEvent]).head.asInstanceOf[CisDeletedEvent].getIds()(0) should equal (ci1.getId)
      listener.eventsPerEventClass(classOf[CisDeletedEvent]).head.asInstanceOf[CisDeletedEvent].getIds()(1) should equal (ci2.getId)
      unsubscribe()
    }

    it("should fire event for move") {
      subscribe()
      val anotherDummyInfrastructure: SampleContainer =
        ciHelper.constructConfigurationItem(classOf[SampleContainer], gen.nextId())
      repository.create(anotherDummyInfrastructure)
      val newId = gen.nextId()
      repository.move(anotherDummyInfrastructure.getId, newId)
      listener.countPerEventClass(classOf[CiMovedEvent]) shouldBe 1

      val ciMovedEvent = listener.eventsPerEventClass(classOf[CiMovedEvent]).head.asInstanceOf[CiMovedEvent]
      anotherDummyInfrastructure.getId should equal (ciMovedEvent.getEntity)
      newId should equal (ciMovedEvent.getNewId)

      repository.delete(newId)
      unsubscribe()
    }

    it("should not fire event when move fails") {
      subscribe()
      val anotherDummyInfrastructure = ciHelper.constructConfigurationItem(classOf[SampleContainer], gen.nextId())
      repository.create(anotherDummyInfrastructure)
      an[IncorrectArgumentException] should be thrownBy {
        repository.move(anotherDummyInfrastructure.getId, "Environments/moved")
      }
      listener.countPerEventClass should not contain classOf[CiMovedEvent]
      unsubscribe()
    }

    it("should fire event for rename") {
      subscribe()
      val anotherDummyInfrastructure = ciHelper.constructConfigurationItem(classOf[SampleContainer], gen.nextId())
      repository.create(anotherDummyInfrastructure)
      val newName = "newName"
      repository.rename(anotherDummyInfrastructure.getId, newName)
      listener.countPerEventClass(classOf[CiRenamedEvent]) shouldBe 1

      val ciRenamedEvent = listener.eventsPerEventClass(classOf[CiRenamedEvent]).head.asInstanceOf[CiRenamedEvent]
      anotherDummyInfrastructure.getId should equal (ciRenamedEvent.getId)
      newName should equal (ciRenamedEvent.getNewName)

      repository.delete(newName)
      unsubscribe()
    }

    it("should fire event for update") {
      subscribe()
      val dummyCi = ciHelper.constructConfigurationItem(classOf[SampleCi],
        gen.nextId().replace("Configuration", "Infrastructure"))
      repository.create(dummyCi)

      dummyCi.setStringProperty("newnicename")
      repository.update(dummyCi)
      listener.countPerEventClass(classOf[CisUpdatedEvent]) shouldBe 1
      listener.eventsPerEventClass(classOf[CisUpdatedEvent]).head
        .asInstanceOf[CisUpdatedEvent].getCis.get(0) should equal (dummyCi)

      repository.delete(dummyCi.getId)
      unsubscribe()
    }

  }

  private def subscribe() {
    listener = new EventCountingListener
    EventBusHolder.register(listener)
  }

  private def unsubscribe() {
    EventBusHolder.deregister(listener)
  }

  class EventCountingListener {
    import scala.collection.mutable.{Map => MMap}
    var totalCount = 0L
    val countPerEventClass = MMap.empty[Class[_], Long]
    val eventsPerEventClass = MMap.empty[Class[_], List[AnyRef]]

    @Subscribe def listen(event: AuditableDeployitEvent) {
      totalCount += 1
      val count = countPerEventClass.getOrElse(event.getClass, 0L)
      countPerEventClass += event.getClass -> (count + 1L)
      val events = eventsPerEventClass.getOrElse(event.getClass, Nil)
      eventsPerEventClass += event.getClass -> (event :: events)
    }
  }
}
