package com.xebialabs.xlplatform.xlrepository.tck.suites

import com.google.common.collect.Sets
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot._
import com.xebialabs.deployit.repository.SearchParameters
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis._
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase

trait RepositorySearchSuite extends RepositorySuiteBase { self: RepositoryFactory[_] =>
  import com.xebialabs.xlplatform.xlrepository.tck.util.CiHelper._

  import collection.convert.wrapAll._

  private var ci: SampleCi = null
  private var otherCi: SampleCi = null
  private var subtypeCi: SampleSubclass = null
  private var parent: Parent = null

  describe("The XL Repository Search Suite") {

    it("should find one ConfigurationItem for specific type") {
      populateData()
      val searchParameters = new SearchParameters().setType(classOf[SampleSubclass])
      val ids = repository.list(searchParameters)
      ids should have size 1
    }

    it("should not find ConfigurationItems for type which is not in the repo") {
      populateData()
      val searchParameters = new SearchParameters().setType(classOf[Resource])
      val ids = repository.list(searchParameters)
      ids should have size 0
    }

    it("should find all ConfigurationItems and return them sorted alphabetically on the node name and not its full id") {
      populateData()
      val searchParameters: SearchParameters = new SearchParameters().setType(classOf[ConfigurationItem])
      val ids = repository.list(searchParameters).map(_.getId)

      ids should have size 12
      val oldCorrectOrder = Seq("Applications", "Configuration", "Environments", "Infrastructure", "Tests", "Configuration/a", "Configuration/a/b1", "Configuration/a/b2", "Configuration/a/b1/c", "Infrastructure/ci", "Infrastructure/otherCi", "Infrastructure/subtypeItem")

      ids should (equal(oldCorrectOrder) or equal(oldCorrectOrder.sorted))
    }

    it("should find three ConfigurationItems for type that also has subtypes") {
      populateData()
      val searchParameters = new SearchParameters().setType(classOf[SampleCi])
      val ids = repository.list(searchParameters)
      ids should have size 3
    }

    it("should find all ConfigurationItems for interface type") {
      populateData()
      val searchParameters = new SearchParameters().setType(classOf[ConfigurationItem])
      val ids = repository.list(searchParameters)
      ids should have size 12
    }

    it("should find one ConfigurationItems with certain property") {
      populateData()
      val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("intProperty", "123")
      val ids = repository.list(params)
      ids should have size 1
      ids.get(0).getId should equal(otherCi.getId)
    }

    it("should find one ConfigurationItem using wildcard suffix") {
      populateData()
      val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("intProperty", "12%")
      val ids = repository.list(params)
      ids should have size 1
      ids.get(0).getId should equal(otherCi.getId)
    }

    it("should find no ConfigurationItem using wrong wildcard search") {
      populateData()
      val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("intProperty", "%4%")
      val ids = repository.list(params)
      ids should have size 0
    }

    it("should find one ConfigurationItem using wildcard prefix") {
      populateData()
      val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("intProperty", "%23")
      val ids = repository.list(params)
      ids should have size 1
      ids.get(0).getId should equal(otherCi.getId)
    }

    it("should find one ConfigurationItem using wildcard around") {
      populateData()
      val params = new SearchParameters().setType(classOf[SampleCi]).addProperty("intProperty", "%1%")
      val ids = repository.list(params)
      ids should have size 1
      ids.get(0).getId should equal(otherCi.getId)
    }

    it("should find ConfigurationItem using wildcard on name") {
      populateData()
      val searchParameters = new SearchParameters().setName("%i")
      val ids = repository.list(searchParameters)
      ids should have size 2
    }

    it("should find ConfigurationItem property reference") {
      populateData()
      val ref = ciHelper.createConfigurationItem(classOf[Referencing], "Infrastructure/ref")
      val referencingCi = ciHelper.constructConfigurationItem(classOf[Referencing], "Infrastructure/referencingCi")
      referencingCi.setCiProperty(ref)
      repository.create(referencingCi)
      try {
        val searchParameters: SearchParameters = new SearchParameters().setType(classOf[Referencing]).addProperty("ciProperty", ref.getId)
        val ids = repository.list(searchParameters)
        ids should have size 1
        ids.get(0).getId should equal(referencingCi.getId)
      } finally {
        repository.delete(referencingCi.getId, ref.getId)
      }
    }


    it("should throw Exception when trying to find set of ConfigurationItem property reference") {
      populateData()
      val referencingCi = ciHelper.constructConfigurationItem(classOf[Referencing], "Infrastructure/referencingCi")
      val ref1 = ciHelper.createConfigurationItem(classOf[Referencing], "Infrastructure/referencingCi1")
      val ref2 = ciHelper.createConfigurationItem(classOf[Referencing], "Infrastructure/referencingCi2")
      referencingCi.setCisProperty(Sets.newHashSet(ref1, ref2))
      repository.create(referencingCi)
      try {
        val searchParameters: SearchParameters = new SearchParameters().setType(classOf[Referencing]).addProperty("cisProperty", ci.getId)
        an[IllegalArgumentException] should be thrownBy {
          repository.list(searchParameters)
        }
      } finally {
        repository.delete(referencingCi.getId, ref1.getId, ref2.getId)
      }
    }

    it("should do case sensitive search on name without wildcard") {
      populateData()
      val searchParameters = new SearchParameters().setName(ci.getName)
      val ids = repository.list(searchParameters)
      ids should have size 1
    }

    it("should do case insensitive search on name wildcard") {
      populateData()
      val searchParameters = new SearchParameters().setName("%ci%")
      val ids = repository.list(searchParameters)
      ids should have size 2
    }

    it("should do case insensitive search on name wildcard with capitals in search") {
      populateData()
      val searchParameters = new SearchParameters().setName("%CI%")
      val ids = repository.list(searchParameters)
      ids should have size 2
    }

    it("should find root ConfigurationItems when setting parent to slash") {
      populateData()
      val ids = repository.list(new SearchParameters().setParent("/")).map(_.getId)
      ids should have size 5
      ids should contain(APPLICATIONS.getRootNodeName)
      ids should contain(ENVIRONMENTS.getRootNodeName)
      ids should contain(INFRASTRUCTURE.getRootNodeName)
      ids should contain(CONFIGURATION.getRootNodeName)
      ids should contain("Tests")
    }

    it("should find two ConfigurationItems when setting parent") {
      populateData()
      val searchParameters = new SearchParameters().setParent(parent.getId)
      val ids = repository.list(searchParameters)
      ids should have size 2
    }
  }

  private def populateData() {
    ci = ciHelper.constructConfigurationItem(classOf[SampleCi], "Infrastructure/ci")
    ci.setStringProperty("aString")
    repository.create(ci)

    otherCi = ciHelper.constructConfigurationItem(classOf[SampleCi], "Infrastructure/otherCi")
    otherCi.setIntProperty(123)
    repository.create(otherCi)

    subtypeCi = ciHelper.constructConfigurationItem(classOf[SampleSubclass], "Infrastructure/subtypeItem")
    subtypeCi.setStringProperty("stringInTheSubtype")
    repository.create(subtypeCi)

    parent = ciHelper.constructConfigurationItem(classOf[Parent], "Configuration/a")
    repository.create(parent)

    val b1 = ciHelper.constructConfigurationItem(classOf[Child], parent.getId + "/b1")
    repository.create(b1)

    val b2 = ciHelper.constructConfigurationItem(classOf[Child], parent.getId + "/b2")
    repository.create(b2)

    val c = ciHelper.constructConfigurationItem(classOf[GrandChild], b1.getId + "/c")
    repository.create(c)
  }
}
