package com.xebialabs.xlplatform.xlrepository.tck.suites

import java.util.{List => JList}

import com.google.common.collect.Lists._
import com.xebialabs.deployit.core._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis._
import com.xebialabs.xlplatform.xlrepository.tck.util.CiHelper._
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase._

import scala.collection.convert.wrapAll._

trait RepositoryTypeSystemSuite extends RepositorySuiteBase {
  self: RepositoryFactory[_] =>

  private val gen = generator.genFor("tck-typesystem")

  describe("The XL Repository TypeSystem Suite") {
    it("should encrypt password property in the repository") {
      val password: Password = ciHelper.constructCi("repo-tck.Password", gen.nextId())
      password.password = "Secret"
      repository.create(password)
      password property "password" should not be (storedInRepository equal "Secret")
      password property "password" shouldBe (storedInRepository startingWith "{b64}")
    }

    it("should not persist/read hidden properties") {
      val hidden: Hidden = ciHelper.constructCi("repo-tck.Hidden", gen.nextId())
      hidden.hidden = "ShowMe!"
      repository.create(hidden)
      val read: Hidden = repository.read(hidden.getId)
      read.hidden shouldEqual "Hidden"
    }

    it("should not persist/read transient properties") {
      val transient: Transient = ciHelper.constructCi("repo-tck.Transient", gen.nextId())
      transient.transient = "PersistMe!"
      repository.create(transient)
      val read: Transient = repository.read(transient.getId)
      read.transient shouldEqual "Transient"
    }

    it("should write List<String> as StringListView") {
      val ci: ListOfString = ciHelper.constructCi("repo-tck.ListOfString", gen.nextId())
      ci.prop = List("foo", "bar")
      repository.create(ci)
      val read: ListOfString = repository.read(ci.getId)
      read.prop shouldBe a[ListOfStringView]
      val view = read.prop.asInstanceOf[ListOfStringView].getWrapped
      view.head should not be 'encrypted
      view.tail.head should not be 'encrypted
    }

    it("should maintain order for a List<String>") {
      val ci: ListOfString = ciHelper.constructCi("repo-tck.ListOfString", gen.nextId())
      ci.prop = List("foo", "bar")
      repository.create(ci)
      val read: ListOfString = repository.read(ci.getId)
      read.prop should contain inOrder("foo", "bar")
      read.prop = read.prop.reverse
      read.prop.add("baz")
      repository.update(read)
      val read1: ListOfString = repository.read(ci.getId, false)
      read1.prop should contain inOrder("bar", "foo", "baz")
    }

    it("should write Set<String> as StringSetView") {
      val ci: SetOfString = ciHelper.constructCi("repo-tck.SetOfString", gen.nextId())
      ci.prop = Set("foo", "bar")
      repository.create(ci)
      val read: SetOfString = repository.read(ci.getId)
      read.prop shouldBe a[SetOfStringView]
      val view = read.prop.asInstanceOf[SetOfStringView].getWrapped
      view.head should not be 'encrypted
      view.tail.head should not be 'encrypted
    }

    it("should write Map<String, String> as MapStringStringView") {
      val ci: MapStringString = ciHelper.constructCi("repo-tck.MapStringString", gen.nextId())
      ci.prop = Map("key" -> "value")
      repository.create(ci)
      val read: MapStringString = repository.read(ci.getId)
      read.prop shouldBe a[MapStringStringView]
      val view = read.prop.asInstanceOf[MapStringStringView].getWrapped
      view.head._2 should not be 'encrypted
    }

    it("should write password List<String> as encrypted StringListView") {
      val ci: PasswordListOfString = ciHelper.constructCi("repo-tck.PasswordListOfString", gen.nextId())
      ci.prop = List("foo", "bar")
      repository.create(ci)
      val read: PasswordListOfString = repository.read(ci.getId)
      read.prop shouldBe a[ListOfStringView]
      val view = read.prop.asInstanceOf[ListOfStringView].getWrapped
      view.head shouldBe 'encrypted
      view.tail.head shouldBe 'encrypted
    }

    it("should write password Set<String> as encrypted StringSetView") {
      val ci: PasswordSetOfString = ciHelper.constructCi("repo-tck.PasswordSetOfString", gen.nextId())
      ci.prop = Set("foo", "bar")
      repository.create(ci)
      val read: PasswordSetOfString = repository.read(ci.getId)
      read.prop shouldBe a[SetOfStringView]
      val view = read.prop.asInstanceOf[SetOfStringView].getWrapped
      view.head shouldBe 'encrypted
      view.tail.head shouldBe 'encrypted
    }

    it("should write password Map<String, String> as encrypted MapStringStringView") {
      val ci: PasswordMapStringString = ciHelper.constructCi("repo-tck.PasswordMapStringString", gen.nextId())
      ci.prop = Map("key" -> "value")
      repository.create(ci)
      val read: PasswordMapStringString = repository.read(ci.getId)
      read.prop shouldBe a[MapStringStringView]
      val view = read.prop.asInstanceOf[MapStringStringView].getWrapped
      view.head._2 shouldBe 'encrypted
    }

    it("should keep partially encrypted SetOfStringView intact") {
      val ci: SetOfString = ciHelper.constructCi("repo-tck.SetOfString", gen.nextId())
      ci.prop = new SetOfStringView()
      ci.prop.asInstanceOf[SetOfStringView].addEncrypted("secret")
      ci.prop.add("plain")
      repository.create(ci)
      val read: SetOfString = repository.read(ci.getId)
      read.prop should have size 2
      read.prop.asInstanceOf[SetOfStringView].getWrapped should contain(new StringValue("plain"))
      read.prop.asInstanceOf[SetOfStringView].getWrapped should contain(new EncryptedStringValue("secret"))
    }

    it("should keep partially encrypted ListOfStringView intact") {
      val ci: ListOfString = ciHelper.constructCi("repo-tck.ListOfString", gen.nextId())
      ci.prop = new ListOfStringView()
      ci.prop.asInstanceOf[ListOfStringView].addEncrypted("secret")
      ci.prop.add("plain")
      repository.create(ci)
      val read: ListOfString = repository.read(ci.getId)
      read.prop should have size 2
      read.prop.asInstanceOf[ListOfStringView].getWrapped should contain(new StringValue("plain"))
      read.prop.asInstanceOf[ListOfStringView].getWrapped should contain(new EncryptedStringValue("secret"))
    }

    it("should keep partially encrypted MapStringStringView intact") {
      val ci: MapStringString = ciHelper.constructCi("repo-tck.MapStringString", gen.nextId())
      ci.prop = new MapStringStringView()
      ci.prop.asInstanceOf[MapStringStringView].put("password", new EncryptedStringValue("secret"))
      ci.prop.put("random", "plain")
      repository.create(ci)
      val read: MapStringString = repository.read(ci.getId)
      read.prop should have size 2
      read.prop.asInstanceOf[MapStringStringView].getWrapped should contain value new StringValue("plain")
      read.prop.asInstanceOf[MapStringStringView].getWrapped should contain value new EncryptedStringValue("secret")
    }

    it("should force encryption of list of string if non-encrypted values stored in repo are read into password property") {
      val ci: PasswordListOfString = ciHelper.constructCi("repo-tck.PasswordListOfString", gen.nextId())
      repository.create(ci)
      setStoredValue(ci, "prop", List("foo", "bar"))

      val read: PasswordListOfString = repository.read(ci.getId)
      read.prop.asInstanceOf[ListOfStringView].getWrapped.foreach {
        _ shouldBe 'encrypted
      }
    }

    it("should force encryption of set of string if non-encrypted values stored in repo are read into password property") {
      val ci: PasswordSetOfString = ciHelper.constructCi("repo-tck.PasswordSetOfString", gen.nextId())
      repository.create(ci)
      setStoredValue(ci, "prop", Set("foo", "bar"))

      val read: PasswordSetOfString = repository.read(ci.getId)
      read.prop.asInstanceOf[SetOfStringView].getWrapped.foreach {
        _ shouldBe 'encrypted
      }
    }

    it("should force encryption of map of string if non-encrypted values stored in repo are read into password property") {
      val ci: PasswordMapStringString = ciHelper.constructCi("repo-tck.PasswordMapStringString", gen.nextId())
      repository.create(ci)
      setStoredValue(ci, "prop", Map("password" -> "secret", "another_secret" -> "hidden"))

      val read: PasswordMapStringString = repository.read(ci.getId)
      read.prop.asInstanceOf[MapStringStringView].getWrapped.values().foreach {
        _ shouldBe 'encrypted
      }
    }

    it("should read and update convert value for property that has changed kind from String to list of string") {
      val ci = ciHelper.constructConfigurationItem(classOf[AllProperties], gen.nextId())
      repository.create(ci)

      setStoredValue(ci, "stringList", "listValue")

      // read it as a different type to test the backwards compatibility of String->ListOfString kind
      var readCi = repository.read[AllProperties](ci.getId)
      readCi.stringList should have size 1
      readCi.stringList.get(0) should equal("listValue")

      readCi.stringList = newArrayList("listValue", "container-by-container")
      repository.update(readCi)

      readCi = repository.read(ci.getId)
      readCi.stringList.size shouldBe 2
      readCi.stringList.get(0) should equal("listValue")
      readCi.stringList.get(1) should equal("container-by-container")
    }

    it("should not fail on absent list of ConfigurationItem containment property") {
      val `type` = Type.valueOf(classOf[AsContainmentList])
      val configurationItem = `type`.getDescriptor.newInstance[AsContainmentList](gen.nextId())
      repository.create(configurationItem)

      removeProperty(configurationItem.getId, "asContainmentList")
      hasProperty(configurationItem.getId, "asContainmentList") shouldBe false

      val read = repository.read[AsContainmentList](configurationItem.getId, false)
      read.getProperty("asContainmentList").asInstanceOf[JList[ConfigurationItem]] should have size 0
    }
  }
}
