package com.xebialabs.xlplatform.xlrepository.tck.suites

import java.util.{List => JList}

import com.google.common.collect.Sets
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot._
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.{ConfigurationItemData, SearchParameters}
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis.SampleCi
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase

import scala.collection.convert.wrapAll._

trait TextSearchSuite extends RepositorySuiteBase {
  self: RepositoryFactory[_] =>

  private val TEST_FOLDER = classOf[TextSearchSuite].getSimpleName
  private val PREFIX = INFRASTRUCTURE.getRootNodeName + "/" + TEST_FOLDER + "/"

  protected def populateRepository() {
    val directory = new Directory
    directory.setId(PREFIX)
    repository.create(directory)
    repository.create(dummyCi("01", "Burgers and coke"))
    repository.create(dummyCi("02", "Burgers and sprite"))
    repository.create(dummyCi("03", "Fish and chips"))
    repository.create(dummyCi("04", "Fish & chips"))
    repository.create(dummyCi("05", "Baked beans with tea"))
    repository.create(dummyCi("06", "Baked tea with beans"))
    repository.create(dummyCi("07", SampleCi.SampleEnum.ONE))
    repository.create(dummyCi("08", SampleCi.SampleEnum.TWO))
  }

  describe("Text search suite") {
    it("should find burgers") {
      populateRepository()
      val ids = searchFor("burgers")

      ids.size shouldBe 2
      ids should contain allOf (ci("01"), ci("02"))
    }

    it("should find burgers without coke") {
      populateRepository()
      val ids = searchFor("burgers -coke")

      ids should have size 1
      ids should contain (ci("02"))
    }

    it("should find burgers and fish") {
      populateRepository()
      val ids = searchFor("burgers OR fish")

      ids should have size 4
      ids should contain allOf (ci("01"), ci("02"), ci("03"), ci("04"))
    }

    it("should find fish and chips") {
      populateRepository()
      val ids = searchFor("fish AND chips")

      ids should have size 1
      ids should contain (ci("03"))
    }

    it("should find all fish") {
      populateRepository()
      val ids = searchFor("fish chips")

      ids should have size 2
      ids should contain allOf (ci("03"), ci("04"))
    }

    it("should find baked tea") {
      populateRepository()
      var ids = searchFor("baked tea")

      ids should have size 2
      ids should contain allOf (ci("05"), ci("06"))

      ids = searchFor("\"baked tea\"")

      ids should have size 1
      ids should contain (ci("06"))
    }

    it("should work on enums") {
      populateRepository()
      val search: SearchParameters = new SearchParameters().addTextSearch("enumProperty", "one OR TWO")
      val ids = repository.list(search)

      ids should have size 2
      ids should contain allOf (ci("07"), ci("08"))
    }

    it("should be able to match several property values") {
      def simpleCi(id: String, propValue: String) = {
        val ci = ciHelper.constructConfigurationItem(classOf[SampleCi], INFRASTRUCTURE.getRootNodeName + "/" + id)
        ci.stringProperty = propValue
        ci
      }
      repository.create(simpleCi("09", "value1"), simpleCi("10", "VALUE1"), simpleCi("11", "value2"), simpleCi("12", "value3"))

      val search = new SearchParameters().addOrPropertiesExpression("stringProperty", Sets.newHashSet("value1", "ValuE2"))
      val ids = repository.list(search).map(_.getId.replace(INFRASTRUCTURE.getRootNodeName + "/", ""))
      ids should have size 3
      ids should contain allOf ("09", "10", "11")
      ids should not contain "12"
    }
  }

  private def searchFor(textQuery: String): JList[ConfigurationItemData] = {
    repository.list(new SearchParameters().addTextSearch("stringProperty", textQuery))
  }

  private def ci(id: String): ConfigurationItemData = {
    new ConfigurationItemData(PREFIX + id, Type.valueOf(classOf[SampleCi]))
  }

  private def dummyCi(id: String, text: String): SampleCi = {
    val ci = ciHelper.constructConfigurationItem(classOf[SampleCi], PREFIX + id)
    ci.setStringProperty(text)
    ci
  }

  private def dummyCi(id: String, value: SampleCi.SampleEnum): SampleCi = {
    val ci = ciHelper.constructConfigurationItem(classOf[SampleCi], PREFIX + id)
    ci.setEnumProperty(value)
    ci
  }
}
