package com.xebialabs.xlplatform.xlrepository.tck.suites

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.Application
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.{ChangeSet, ItemAlreadyExistsException, SearchParameters}
import com.xebialabs.license.{AmountOfCisExceededException, LicenseRepositoryIdException}
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.util.{IdGenerator, RepositorySuiteBase}

trait RepositoryLicenseEnforcementSuite extends RepositorySuiteBase {
  self: RepositoryFactory[_] =>

  import com.xebialabs.xlplatform.xlrepository.tck.util.CiHelper._

  import collection.JavaConverters._

  val gen = new IdGenerator(ConfigurationItemRoot.APPLICATIONS).genFor("license-")

  val twoAppLicense =
    """--- License ---
      |License version: 3
      |Product: XL Deploy
      |Licensed to: Roadrunner Inc.
      |Contact: Jane Doe
      |Repository id: 123456789
      |Expires after: 2099-01-31
      |Maximum number of users: 20
      |Licensed number of CIs: udm.Application=2
      |Licensed number of CIs: overthere.Host=1
      |Licensed plugin: yak-plugin
      |Licensed plugin: was-plugin
      |Support policy: 24x7
      |Edition: Enterprise
      |--- Signature (SHA1withDSA) ---
      |302c02144af1175153664c093d4995dc9fac3cd5a9cd81dd02144db23c2460de3d6a6259976c0480
      |305f89f317cf
      |--- End of Signature ---""".stripMargin('|')

  describe("The XL Repository license enforcement suite") {
    it("should throw exception if the limit of CIs is reached when creating") {
      withLicense(twoAppLicense) {
        repository.create(newApp, newApp)
        val ex = intercept[AmountOfCisExceededException] {
          repository.create(newApp)
        }
        ex.getMessage shouldEqual "Unable to create udm.Application. Your license is limited to 2 udm.Application CIs and you currently have 2."
      }
    }

    it("should throw an exception if the limit is reached while creating multiple CIs") {
      withLicense(twoAppLicense) {
        an[AmountOfCisExceededException] shouldBe thrownBy {
          repository.create(newApp, newApp, newApp)
        }
        repository.list(new SearchParameters().setType(classOf[Application])) shouldBe empty
      }
    }

    it("should not throw an exception if max CIs are stored and one is updated via createOrUpdate") {
      withLicense(twoAppLicense) {
        val app1 = newApp
        repository.create(app1, newApp)
        val ci: Application = ciHelper.constructCi(classOf[Application], app1.getId)
        repository.createOrUpdate(ci)
      }
    }

    it("should throw an exception if max CIs are stored and one is created via createOrUpdate") {
      withLicense(twoAppLicense) {
        repository.create(newApp, newApp)
        an[AmountOfCisExceededException] shouldBe thrownBy {
          repository.createOrUpdate(newApp)
        }
      }
    }

    it("should not count copy failure operation for the maximum number of CIs") {
      withLicense(twoAppLicense) {
        val app = newApp
        val dir: Directory = ciHelper.constructCi(classOf[Directory], gen.nextId())
        repository.create(app, dir)

        an[Exception] shouldBe thrownBy {
          // Expected this one
          repository.copy(app.getId, dir.getId)
        }

        // Should still be possible
        repository.create(newApp)
      }
    }

    it("should not count creation failure for the maximum number of CIs") {
      withLicense(twoAppLicense) {
        val app = newApp
        repository.create(app)

        an[ItemAlreadyExistsException] shouldBe thrownBy {
          repository.create(app)
        }

        repository.create(newApp)
      }
    }

    it("should throw an exception if copying a CI leads to more than the maximum allowed number of CIs") {
      withLicense(twoAppLicense) {
        val app1 = newApp
        repository.create(app1, newApp)
        an[AmountOfCisExceededException] shouldBe thrownBy {
          repository.copy(app1.getId, gen.nextId())
        }
      }
    }

    it("should throw exception when copying a tree of CIs which has a CI type which exceeds the maximum set") {
      withLicense(twoAppLicense) {
        val dir: Directory = ciHelper.constructCi(classOf[Directory], gen.nextId())
        val subdir:Directory = ciHelper.constructCi(classOf[Directory], gen.subId(dir))
        val app1:Application = ciHelper.constructCi(classOf[Application], gen.subId(subdir))
        val app2:Application = ciHelper.constructCi(classOf[Application], gen.subId(subdir))
        repository.create(dir, subdir, app1, app2)

        an[AmountOfCisExceededException] shouldBe thrownBy {
          repository.copy(dir.getId, gen.nextId())
        }
      }
    }

    it("should throw an exception when execution of a changeset leads to more than the maximum number of CIs") {
      withLicense(twoAppLicense) {
        val set: ChangeSet = new ChangeSet()
        set.create(List(newApp, newApp, newApp).asJava)

        an[AmountOfCisExceededException] shouldBe thrownBy {
          repository.execute(set)
        }
      }
    }

    describe("for repository ids") {
      it("should store the repository id") {
        getRepositoryId shouldEqual null
        metadataService.validateAndStoreRepositoryId("my-repo-id")
        getRepositoryId should not equal null
      }

      it("should not be possible to overwrite the repository id with a different value") {
        metadataService.validateAndStoreRepositoryId("my-repo-id")
        val ex = intercept[LicenseRepositoryIdException] {
          metadataService.validateAndStoreRepositoryId("another-id")
        }
        ex.getMessage shouldEqual "New repository id 'another-id' is different from current repository id"
      }

      it("should be possible to overwrite the repository id with the same value") {
        metadataService.validateAndStoreRepositoryId("my-repo-id")
        metadataService.validateAndStoreRepositoryId("my-repo-id")
      }

      it("should be possible to validate a 'null' repository id, while keeping the repository locked to the original id") {
        // This prevents users from temporarily using a 'null' license (in whatever way they obtain it), and then requesting a new trial license again.
        metadataService.validateAndStoreRepositoryId("my-repo-id")
        val storedRepoId = getRepositoryId
        metadataService.validateAndStoreRepositoryId(null)
        getRepositoryId shouldEqual storedRepoId
      }
    }
  }

  def newApp: Application = {
    ciHelper.constructCi(Type.valueOf(classOf[Application]), gen.nextId())
  }
}
