package com.xebialabs.xlplatform.xlrepository.tck.suites

import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis._
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase

import scala.collection.JavaConverters._

trait RepositoryStructuralIntegritySuite extends RepositorySuiteBase {
  self: RepositoryFactory[_] =>

  import com.xebialabs.xlplatform.xlrepository.tck.util.CiHelper._

  private val gen = generator.genFor("tck-structint")
  private val testGen = testsRootGenerator.genFor("tck-structint")

  describe("The XL Repository Structural Integrity Suite") {
    it("should not store a ConfigurationItem under the wrong root") {
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", "Applications/wrong-root")
      an[IncorrectArgumentException] should be thrownBy {
        repository.create(ci)
      }
      ci shouldNot existInRepository
    }

    it("should not store a ConfigurationItem under a non-root") {
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", "no-root")
      an[IncorrectArgumentException] should be thrownBy {
        repository.create(ci)
      }
      ci shouldNot existInRepository
    }

    it("should not store a child ConfigurationItem under a root") {
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Child", gen.nextId())
      an[IncorrectArgumentException] should be thrownBy {
        repository.create(ci)
      }
      ci shouldNot existInRepository
    }

    it("should not store a child ConfigurationItem under a non-root") {
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Child", "no-root")
      an[IncorrectArgumentException] should be thrownBy {
        repository.create(ci)
      }
      ci shouldNot existInRepository
    }

    // modeshape implementation is more lenient so this fails
    ignore("should not store a rooted ConfigurationItem as a nested ConfigurationItem") {
      val correct: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", gen.nextId())
      repository.create(correct)
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Basic", gen.subId(correct))
      an[IncorrectArgumentException] should be thrownBy {
        repository.create(ci)
      }
      ci shouldNot existInRepository
    }

    it("should store a child ConfigurationItem as a nested ConfigurationItem") {
      val correct: ConfigurationItem = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      repository.create(correct)
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Child", gen.subId(correct))
      repository.create(ci)
      ci should existInRepository
    }

    it("should store a child subclass ConfigurationItem as a nested ConfigurationItem") {
      val correct: ConfigurationItem = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      repository.create(correct)
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.ChildSub", gen.subId(correct))
      repository.create(ci)
      ci should existInRepository
    }

    it("should store a child ConfigurationItem as a nested ConfigurationItem under a parent subclass") {
      val correct: ConfigurationItem = ciHelper.constructCi("repo-tck.ParentSub", gen.nextId())
      repository.create(correct)
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.Child", gen.subId(correct))
      repository.create(ci)
      ci should existInRepository
      ci should have
    }

    it("should read/write CIs with circular references") {
      val correct: ConfigurationItem = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      repository.create(correct)
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.ChildWithKnownParent", gen.subId(correct))
      repository.create(ci)
      val read: ChildWithKnownParent = repository.read(ci.getId)
      read.parent should not equal null
      read.parent.myChildren should have size 1
      read.parent.myChildren should contain(read)
    }

    it("should not store a child ConfigurationItem nested under a non-matching parent ConfigurationItem") {
      val correct: ConfigurationItem = ciHelper.constructCi("repo-tck.Parent", gen.nextId())
      repository.create(correct)
      val ci: ConfigurationItem = ciHelper.constructCi("repo-tck.IsNotAChild", gen.subId(correct))
      an[IncorrectArgumentException] should be thrownBy {
        repository.create(ci)
      }
      ci shouldNot existInRepository
    }

    it("should store directory ci with id ending with '/' character") {
      def createDir(id: String) = Type.valueOf(classOf[Directory]).getDescriptor.newInstance[Directory](id)
      val ci: ConfigurationItem = createDir(gen.nextId())
      repository.create(ci)
      val ci2: ConfigurationItem = createDir(ci.getId + "/something/" )
      repository.create(ci2)
      val read: ConfigurationItem = repository.read(ci2.getId)
      ci2 should existInRepository
      ci2.getId should be(read.getId)
      ci2.getName should be(read.getName)
    }

    describe("list of contained ConfigurationItems") {
      it("should be automatically updated when a new child is stored") {
        val correct: OrderedParent = ciHelper.constructCi("repo-tck.OrderedParent", gen.nextId())
        repository.create(correct)
        val kid1: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(correct))
        repository.create(kid1)
        val kid2: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(correct))
        repository.create(kid2)
        val read: OrderedParent = repository.read(correct.getId)
        read.offspring.get(0).getId == kid1.getId
        read.offspring.get(1).getId == kid2.getId
      }

      it("should be stored in specific order") {
        val correct: OrderedParent = ciHelper.constructCi("repo-tck.OrderedParent", gen.nextId())
        val kid1: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(correct))
        val kid2: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(correct))
        correct.offspring.add(kid2)
        correct.offspring.add(kid1)
        repository.create(correct, kid1, kid2)
        val read: OrderedParent = repository.read(correct.getId)
        read.offspring.get(0).getId == kid2.getId
        read.offspring.get(1).getId == kid1.getId
      }

      it("should remove from the list when a child is removed") {
        val correct: OrderedParent = ciHelper.constructCi("repo-tck.OrderedParent", gen.nextId())
        repository.create(correct)
        val kid1: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(correct))
        repository.create(kid1)
        repository.delete(kid1.getId)
        val read: OrderedParent = repository.read(correct.getId)
        read.offspring shouldBe 'empty
      }

      it("should only add contained ConfigurationItems to the list") {
        val correct: OrderedParent = ciHelper.constructCi("repo-tck.OrderedParent", gen.nextId())
        val correct2: OrderedParent = ciHelper.constructCi("repo-tck.OrderedParent", gen.nextId())
        repository.create(correct, correct2)
        val kid: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(correct2))
        an[Exception] should be thrownBy {
          correct.offspring.add(kid)
          repository.update(correct)
        }
      }

      it("should not remove contained ConfigurationItems from the repository when they're removed from the list") {
        val correct: OrderedParent = ciHelper.constructCi("repo-tck.OrderedParent", gen.nextId())
        val kid1: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(correct))
        val kid2: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(correct))
        repository.create(correct, kid1, kid2)
        val read: OrderedParent = repository.read(correct.getId)
        read.offspring should have size 2
        read.offspring.get(0).getId shouldEqual kid1.getId
        read.offspring.remove(0)
        repository.update(read)
        val read1: OrderedParent = repository.read(correct.getId, false)
        read1.offspring should have size 2
        read1.offspring.get(0).getId shouldEqual kid2.getId
      }

      it("should update to a new order") {
        val parent: OrderedParent = ciHelper.constructCi("repo-tck.OrderedParent", gen.nextId())
        val kid1: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
        val kid2: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
        parent.offspring = List(kid1, kid2).asJava
        repository.create(parent, kid1, kid2)
        val read: OrderedParent = repository.read(parent.getId)
        read.offspring should have size 2
        read.offspring.get(0).getId shouldEqual kid1.getId
        read.offspring = read.offspring.asScala.reverse.asJava
        repository.update(read)
        val read1: OrderedParent = repository.read(parent.getId, false)
        read1.offspring should have size 2
        read1.offspring.get(0).getId shouldEqual kid2.getId
      }

      it("should not add the child a second time when it is updated") {
        val parent: OrderedParent = ciHelper.constructCi("repo-tck.OrderedParent", gen.nextId())
        val kid1: Child = ciHelper.constructCi("repo-tck.Child", gen.subId(parent))
        repository.create(parent, kid1)
        repository.update(kid1)
        val read: OrderedParent = repository.read(parent.getId)
        read.offspring should have size 1
        read.offspring.get(0).getId shouldEqual kid1.getId
      }
    }

    describe("directories") {
      it("should create and delete a directory") {
        val directory: Directory = ciHelper.createDirectory(gen.nextId())
        repository.exists(directory.getId) shouldEqual true
        repository.delete(directory.getId)
        repository.exists(directory.getId) shouldEqual false
      }

      it("should not create a directory at the root") {
        an[IncorrectArgumentException] shouldBe thrownBy {
          repository.create(ciHelper.constructConfigurationItem(classOf[Directory], "dir"))
        }
      }

      it("should create nested directory structures") {
        val directory: Directory = ciHelper.createDirectory(gen.nextId())
        val subDir: Directory = ciHelper.createDirectory(gen.subId(directory))
        repository.exists(subDir.getId) shouldEqual true
      }

      it("should not store nested configuration items in a directory") {
        val directory: Directory = ciHelper.createDirectory(gen.nextId())
        val ci: Child = ciHelper.constructCi(classOf[Child], gen.subId(directory))
        an[IncorrectArgumentException] shouldBe thrownBy {
          repository.create(ci)
        }
        repository.exists(ci.getId) shouldEqual false
      }

      it("should store a rooted configuration item in a directory (under the correct root)") {
        val directory: Directory = ciHelper.createDirectory(gen.nextId())
        val ci: Parent = ciHelper.constructCi(classOf[Parent], gen.subId(directory))
        repository.create(ci)
        repository.exists(ci.getId) shouldEqual true
      }

      it("should not store a rooted configuration item in a directory under the incorrect root") {
        val directory: Directory = ciHelper.createDirectory(gen.nextId())
        val ci: OtherRoot = ciHelper.constructCi(classOf[OtherRoot], gen.subId(directory))
        an[IncorrectArgumentException] shouldBe thrownBy {
          repository.create(ci)
        }
        repository.exists(ci.getId) shouldEqual false
      }

      it("should store a rooted configuration under non-standard root") {
        val directory: Directory = ciHelper.createDirectory(testGen.nextId())
        val ci: RootByName = ciHelper.constructCi(classOf[RootByName], testGen.subId(directory))
        repository.create(ci)
        repository.exists(ci.getId) shouldEqual true
      }

      it("should not store a rooted configuration item when the parent dirs have not been created") {
        val dir: Directory = ciHelper.constructConfigurationItem(classOf[Directory], gen.nextId())
        val ci: Parent = ciHelper.constructCi(classOf[Parent], gen.subId(dir))
        an[Exception] shouldBe thrownBy {
          repository.create(ci)
        }
        repository.exists(ci.getId) shouldEqual false
      }
    }
  }
}
