package com.xebialabs.xlplatform.xlrepository.tck.suites

import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis.{Phase, Referencer, Release}
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase

trait RepositoryReferentialIntegritySuite extends RepositorySuiteBase { self: RepositoryFactory[_] =>
  import com.xebialabs.xlplatform.xlrepository.tck.util.CiHelper._

  private val gen = generator.genFor("tck-refint")

  describe("The XL Repository Referential Integrity Suite") {
    it("should not create a ConfigurationItem with an invalid reference") {
      val referencer: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())
      val referenced: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())
      referencer.ref = referenced
      an[Exception] should be thrownBy {
        repository.create(referencer)
      }
    }

    it("should persist a ConfigurationItem with its references") {
      val referencer: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())
      val referenced: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())
      referencer.ref = referenced
      repository.create(referenced, referencer)
      referenced should existInRepository
      referencer should existInRepository
    }

    it("should persist a ConfigurationItem with it references (reverse creation order)") {
      val referencer: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())
      val referenced: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())
      referencer.ref = referenced
      repository.create(referencer, referenced)
      referenced should existInRepository
      referencer should existInRepository

    }

    it("should not be able to delete a referenced ConfigurationItem") {
      val referencer: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())
      val referenced: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())
      referencer.ref = referenced
      repository.create(referenced, referencer)
      val ex = intercept[ItemInUseException] {
        repository.delete(referenced.getId)
      }
      ex.getMessage should include(referencer.getId)
      referenced should existInRepository
    }

    it("should correctly handle circular references and update parent ConfigurationItem ") {
      val release: Release = ciHelper.constructCi("repo-tck.Release", gen.nextId())
      repository.create(release)

      val phase: Phase = ciHelper.constructCi("repo-tck.Phase", release.getId +"/Phase1")
      phase.release = release
      repository.create(phase)

      val rel = repository.read[Release](release.getId)
      rel.phase should not be null
      rel.phase should have size 1
      rel should equal(rel.phase.get(0).release)
    }

    it("should update references when renaming referenced ConfigurationItem") {
      val referencer: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())
      val referenced: Referencer = ciHelper.constructCi("repo-tck.Referencer", gen.nextId())

      referencer.ref = referenced
      repository.create(referenced, referencer)

      val newId = gen.nextId()
      repository.move(referenced.getId, newId)

      val updatedReferencer = repository.read[Referencer](referencer.getId)
      updatedReferencer should not be null
      updatedReferencer.ref should not be null
      updatedReferencer.ref.getId shouldBe newId
    }

  }

}
