package com.xebialabs.xlplatform.xlrepository.tck.suites

import com.xebialabs.deployit.engine.spi.command._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.{Application, ConfigurationItem, Environment}
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.{ItemAlreadyExistsException, RepositoryEventListener, RepositoryServiceHolder}
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase

import java.text.MessageFormat
import java.util.Collections

trait UniquenessValidationSuite extends RepositorySuiteBase {
  self: RepositoryFactory[_] =>

  private var eventListener: RepositoryEventListener = _

  describe("Application Uniqueness validator") {
    it("should not create application with duplicated names") {
      before()
      eventListener.receiveCreate(new CreateCiCommand(constructApp("1")))
      intercept[ItemAlreadyExistsException] {
        eventListener.receiveCreate(new CreateCiCommand(constructApp("1")))
      }.getMessage shouldBe "Application with name [1] already exists"
    }

    it("should not create applications with duplicated names") {
      eventListener.receiveCreate(new CreateCisCommand(Collections.singletonList(constructApp("1"))))
      intercept[ItemAlreadyExistsException] {
        eventListener.receiveCreate(new CreateCisCommand(Collections.singletonList(constructApp("1"))))
      }.getMessage shouldBe "Application with name [1] already exists"
    }

    it("should not allow to copy application with duplicated names") {
      val app1 = constructApp("1")
      val copyApp1 = constructApp("Copy 1")
      intercept[ItemAlreadyExistsException] {
        eventListener.receiveCreate(new CreateCiCommand(app1))
        eventListener.receiveCreate(new CreateCiCommand(copyApp1))
        eventListener.receiveCopy(new CopyCiCommand(app1.getId, copyApp1.getId, app1.getType))
      }.getMessage shouldBe "Application with name [Copy 1] already exists"
    }

    it("should not allow to copy application if application name is unique") {
      val app1 = constructApp("1")
      val copyApp1 = constructApp("Copy 1")
      eventListener.receiveCreate(new CreateCiCommand(app1))
      eventListener.receiveCopy(new CopyCiCommand(app1.getId, copyApp1.getId, app1.getType))
    }

    it("should not allow to rename application to existed name") {
      val app1 = constructApp("1")
      val app2 = constructApp("2")
      intercept[ItemAlreadyExistsException] {
        eventListener.receiveCreate(new CreateCiCommand(app1))
        eventListener.receiveCreate(new CreateCiCommand(app2))
        eventListener.receiveRename(new RenameCiCommand(app1.getId, app2.getName, app1.getType))
      }.getMessage shouldBe "Application with name [2] already exists"
    }

    it("should allow to rename application if application name is unique") {
      val app1 = constructApp("1")
      val app2 = constructApp("2")
      val app3 = constructApp("3")
      eventListener.receiveCreate(new CreateCiCommand(app1))
      eventListener.receiveCreate(new CreateCiCommand(app2))
      eventListener.receiveRename(new RenameCiCommand(app1.getId, app3.getName, app1.getType))
    }

    it("should allow to update created ConfigurationItem") {
      val app1 = constructApp("update-1")
      eventListener.receiveCreate(new CreateCiCommand(app1))
      eventListener.receiveUpdate(new UpdateCiCommand(app1, app1))
    }

    it("should allow to create application and environment with the same names") {
      val app1 = constructApp("2")
      val env1 = constructEnv("2")
      eventListener.receiveCreate(new CreateCiCommand(app1))
      eventListener.receiveCreate(new CreateCiCommand(env1))
    }

    it("should ensure child application names are unique when copying simple directory") {
      eventListener.receiveCreate(new CreateCiCommand(constructDirectory("apps")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps/0")))

      eventListener.receiveCopy(new CopyCiCommand("Applications/apps", "Applications/apps Copy", Type.valueOf(classOf[Directory])))

      exists("apps Copy")
      exists("apps Copy/0 Copy")
    }

    it("should ensure child application names are unique when copying nested directory") {
      eventListener.receiveCreate(new CreateCiCommand(constructDirectory("apps")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps/0")))
      eventListener.receiveCreate(new CreateCiCommand(constructDirectory("apps/pet")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps/pet/1")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps/pet/2")))

      eventListener.receiveCopy(new CopyCiCommand("Applications/apps", "Applications/apps Copy", Type.valueOf(classOf[Directory])))

      exists("apps Copy")
      exists("apps Copy/0 Copy")
      exists("apps Copy/pet")
      exists("apps Copy/pet/1 Copy")
      exists("apps Copy/pet/2 Copy")
    }

    it("should ensure child application names are unique when copying already copied directory") {
      eventListener.receiveCreate(new CreateCiCommand(constructDirectory("apps")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps/0")))
      eventListener.receiveCreate(new CreateCiCommand(constructDirectory("apps/pet")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps/pet/1")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps/pet/2")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps/pet/3")))

      eventListener.receiveCreate(new CreateCiCommand(constructDirectory("apps-other")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps-other/0-other")))
      eventListener.receiveCreate(new CreateCiCommand(constructDirectory("apps-other/pet")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps-other/pet/1 Copy")))
      eventListener.receiveCreate(new CreateCiCommand(constructApp("apps-other/pet/2 Copy Copy")))

      eventListener.receiveCopy(new CopyCiCommand("Applications/apps", "Applications/apps Copy", Type.valueOf(classOf[Directory])))

      exists("apps Copy")
      exists("apps Copy/0 Copy")
      exists("apps Copy/pet")
      exists("apps Copy/pet/1 Copy Copy")
      exists("apps Copy/pet/2 Copy")
      exists("apps Copy/pet/3 Copy")
    }
  }

  def before(): Unit = {
    new RepositoryServiceHolder(repository)
    eventListener = new RepositoryEventListener
  }

  private def constructApp(id: String): ConfigurationItem = {
    Type.valueOf(classOf[Application]).getDescriptor.newInstance[Application]("Applications/" + id)
  }

  private def exists(id: String): Unit = {
    repository.exists("Applications/" + id) shouldBe true
  }

  private def constructDirectory(id: String): ConfigurationItem = {
    Type.valueOf(classOf[Directory]).getDescriptor.newInstance[Directory]("Applications/" + id)
  }

  private def constructEnv(id: String): ConfigurationItem = {
    val env = new Environment
    env.setId(MessageFormat.format("Environments/{0}", id))
    env
  }
}
