package com.xebialabs.xlplatform.xlrepository.tck.util

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot
import com.xebialabs.deployit.repository.{SearchParameters, WorkDirFactory}
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.xlplatform.test.TemporaryDirectory
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.util.CiHelper._
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase.InRepository
import grizzled.slf4j.Logging
import org.scalatest.enablers.Existence
import org.scalatest.funspec.AnyFunSpecLike
import org.scalatest.matchers.should.Matchers
import org.scalatest.matchers.{BeMatcher, MatchResult, Matcher}
import org.springframework.security.authentication.TestingAuthenticationToken
import org.springframework.security.core.context.SecurityContextHolder

import java.io.File
import scala.jdk.CollectionConverters._

trait RepositorySuiteBase extends AnyFunSpecLike with Matchers with RepositorySugar with Logging with TemporaryDirectory {
  self: RepositoryFactory[_] =>
  val generator = new IdGenerator(ConfigurationItemRoot.CONFIGURATION)
  val testsRootGenerator = new IdGenerator("Tests")

  val workDirFactory: WorkDirFactory = new WorkDirFactory("tck-work")

  override protected def beforeEach(): Unit = {
    super.beforeEach()
    workDirFactory.postConstruct()
    SecurityContextHolder.getContext.setAuthentication(new TestingAuthenticationToken("foo", "bar"))
  }

  override protected def afterEach(): Unit = {
    val ids = repository.list(new SearchParameters().setType("udm.BaseConfigurationItem")).asScala.filterNot(_.getType == stringToType("internal.Root")).map(_.getId).distinct
    debug(s"Deleting $ids")
    repository.delete(ids.toSeq :_*)
    deleteDir(new File(workDirFactory.getBaseWorkDirPath))
    SecurityContextHolder.getContext.setAuthentication(null)
    super.afterEach()

  }

  case object ExistsMatcher extends Matcher[AnyRef] {
    override def apply(left: AnyRef): MatchResult = {
      val id = left match {
        case ci: ConfigurationItem => ci.getId
        case id: String => id
        case x@_ => throw new IllegalArgumentException(s"Cannot check existence for $x")
      }
      MatchResult(repository.exists(id),
        s"$id does not exist in the repository",
        s"$id exists in the repository when it shouldn't have")
    }
  }

  def existInRepository: ExistsMatcher.type = ExistsMatcher

  def storedInRepository = new InRepository(this)
}

object RepositorySuiteBase {
  case class PropertyValue(ci: ConfigurationItem, propertyName: String)

  case class PropertyValueInRepositoryMatcher(value: AnyRef) extends Matcher[PropertyValue] {
    override def apply(left: PropertyValue): MatchResult = {
      null
    }
  }

  implicit class RichCi(val ci: ConfigurationItem) extends AnyVal {
    def property(name: String): PropertyValue = PropertyValue(ci, name)
  }

  final class InRepository(repository: RepositoryFactory[_]) {
    def equal(right: Any): BeMatcher[PropertyValue] = (left: PropertyValue) => {
      val v: AnyRef = repository.getStoredValue(left.ci, left.propertyName)
      new MatchResult(v == right, s"$v and $right were not equal", s"$v and $right were equal")
    }

    def startingWith(right: String): BeMatcher[PropertyValue] = (left: PropertyValue) => {
      val v = repository.getStoredValue[String](left.ci, left.propertyName)
      new MatchResult(v.startsWith(right), s"$v does not start with $right", s"$v starts with $right")
    }
  }

  implicit def existenceOfFile[FILE <: OverthereFile]: Existence[FILE] = (file: FILE) => file.exists

}
