package com.xebialabs.xlplatform.xlrepository.tck.suites

import com.xebialabs.deployit.exception.{UnknownLabelException, UnknownRevisionException}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis._
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase
import org.springframework.security.authentication.TestingAuthenticationToken
import org.springframework.security.core.context.SecurityContextHolder

trait HistorySuite extends RepositorySuiteBase {
  self: RepositoryFactory[_] =>

  private val VERSIONED_NODE_PROPERTY_VALUE_1 = "the first value"
  private val VERSIONED_NODE_PROPERTY_VALUE_2 = "the second value"
  private val gen = generator.genFor("tck-hist")

  describe("XL Repository Version history test") {

    it("should get version revisions") {
      val entity = ciWithVersionedNodeProp(gen.nextId())
      try {
        entity.stringProperty = VERSIONED_NODE_PROPERTY_VALUE_2
        repository.update(entity)
        entity.stringProperty = VERSIONED_NODE_PROPERTY_VALUE_1
        repository.update(entity)
        val entities = history.getVersionRevisions(entity.getId)
        entities should not be null
        entities should have size 3
      } finally repository.delete(entity.getId)
    }

    it("should not get version revisions for unversioned ConfigurationItem") {
      val entity = ciHelper.constructConfigurationItem(classOf[Unversioned], gen.nextId())
      entity.setStringProperty(VERSIONED_NODE_PROPERTY_VALUE_1)
      repository.create(entity)
      try {
        entity.setStringProperty(VERSIONED_NODE_PROPERTY_VALUE_2)
        repository.update(entity)
        entity.setStringProperty(VERSIONED_NODE_PROPERTY_VALUE_1)
        repository.createOrUpdate(entity)
        val entities = history.getVersionRevisions(entity.getId)
        entities should not be null
        entities should have size 1
      } finally {
        repository.delete(entity.getId)
      }
    }

    it("should get revisions for renamed nodes") {
      val entity = ciWithVersionedNodeProp(gen.nextId())

      entity.setStringProperty(VERSIONED_NODE_PROPERTY_VALUE_2)
      repository.update(entity)

      val newEntityId = entity.getId + "-renamed"
      repository.rename(entity.getId, entity.getName + "-renamed")

      val entities = history.getVersionRevisions(newEntityId)
      entities should not be null
      entities should have size 2

      repository.delete(newEntityId)
    }

    it("should get version history for moved nodes") {
      val group1 = ciHelper.createConfigurationItem(classOf[Directory], gen.nextId())
      val group2 = ciHelper.createConfigurationItem(classOf[Directory], gen.nextId())
      val entity = ciWithVersionedNodeProp(group1.getId + "/entity")

      entity.setStringProperty(VERSIONED_NODE_PROPERTY_VALUE_2)
      repository.update(entity)

      val newEntityId = group2.getId + "/entity"
      repository.move(entity.getId, newEntityId)

      val entities = history.getVersionRevisions(newEntityId)
      entities should not be null
      entities should have size 2

      repository.delete(group1.getId)
      repository.delete(group2.getId)
    }

    it("should get revision of ConfigurationItem") {
      val wVersion = ciWithVersionedNodeProp(gen.nextId())
      wVersion.setStringProperty(VERSIONED_NODE_PROPERTY_VALUE_2)
      repository.update(wVersion)

      val versionRevisions = history.getVersionRevisions(wVersion.getId)
      val configurationItemRevision = versionRevisions.get(0)
      val configurationItem = history.readRevision[Versioned](wVersion.getId, configurationItemRevision.getRevisionId)
      configurationItem.getId should equal (wVersion.getId)
      wVersion.getStringProperty should equal (VERSIONED_NODE_PROPERTY_VALUE_2)
      configurationItem.getStringProperty should equal (VERSIONED_NODE_PROPERTY_VALUE_1)
    }

    it("should throw exception when requesting unknown revision") {
      val versionProp = ciWithVersionedNodeProp(gen.nextId())
      val versionRevisions = history.getVersionRevisions(versionProp.getId)
      versionRevisions should have size 1
      try intercept[UnknownRevisionException] {
        history.readRevision(versionProp.getId, "100.0")
      } finally repository.delete(versionProp.getId)
    }

    it("should delete version history when created item is deleted") {
      val entity = ciWithVersionedNodeProp(gen.nextId())
      val (versionsBefore, ctx) = getNumberOfVersions(entity.getId)
      versionsBefore === 1L
      repository.delete(entity.getId)
      val (versionsAfter, _) = getNumberOfVersions(entity.getId, ctx)
      versionsAfter === -1L
    }

    it("should delete version history when updated item is deleted") {
      val entity = ciWithVersionedNodeProp(gen.nextId())

      val (versions1, ctx1) = getNumberOfVersions(entity.getId)
      versions1 === 1L

      entity.setStringProperty(VERSIONED_NODE_PROPERTY_VALUE_2)
      repository.update(entity)

      val (versions2, ctx2) = getNumberOfVersions(entity.getId, ctx1)
      versions2 === 2L

      repository.delete(entity.getId)

      val (versions3, _) = getNumberOfVersions(entity.getId, ctx2)
      versions3 === -1L
    }

    it("should delete version history of nested elements when item is deleted") {
      val entity = ciHelper.createConfigurationItem(classOf[Parent], gen.nextId())
      val (versions1, ctx1) = getNumberOfVersions(entity.getId)
      versions1 === 1L

      val childEntity = ciHelper.createConfigurationItem(classOf[Child], entity.getId + "/child")
      val (versions2, ctx2) = getNumberOfVersions(childEntity.getId)
      versions2 === 1L

      repository.delete(entity.getId)

      val (versions11, _) = getNumberOfVersions(entity.getId, ctx1)
      versions11 === -1

      val (versions22, _) = getNumberOfVersions(entity.getId, ctx2)
      versions22 === -1
    }

    it("should store user with created and updated nodes") {
      def setCredentials(alice: String, foo: String) {
        SecurityContextHolder.getContext.setAuthentication(new TestingAuthenticationToken(alice, foo))
      }
      setCredentials("alice", "foo")
      val entity = ciWithVersionedNodeProp(gen.nextId())

      setCredentials("bob", "foo")
      entity.setStringProperty(VERSIONED_NODE_PROPERTY_VALUE_2)
      repository.update(entity)

      setCredentials("mallory", "foo")
      entity.setStringProperty(VERSIONED_NODE_PROPERTY_VALUE_1)
      repository.update(entity)

      val versionRevisions = history.getVersionRevisions(entity.getId)
      versionRevisions.get(0).getUsername should equal ("alice")
      versionRevisions.get(1).getUsername should equal ("bob")
      SecurityContextHolder.getContext.setAuthentication(null)
    }

    it("should work for deployeds") {
      val container = ciHelper.createContainer("container", classOf[SampleContainer])
      val ci = ciHelper.constructConfigurationItem(classOf[SampleDeployedProperty], container.getId + "/conf")
      ci.setContainer(container)
      ci.setProperty("stringProperty", "Ben")
      repository.create(ci)
      val read = repository.read[SampleDeployedProperty](ci.getId)
      read.setProperty("stringProperty", "Betty")
      repository.update(read)

      val versionRevisions = history.getVersionRevisions(read.getId)
      history.readRevision[ConfigurationItem](read.getId, versionRevisions.get(0).getRevisionId)
    }
  }

  private def ciWithVersionedNodeProp(id: String): Versioned = {
    val ci = ciHelper.constructConfigurationItem(classOf[Versioned], id)
    ci.stringProperty = VERSIONED_NODE_PROPERTY_VALUE_1
    repository.create(ci)
    ci
  }
}