package com.xebialabs.xlplatform.xlrepository.tck.suites

import java.io.{File, FileOutputStream, PrintWriter}

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.{ChangeSet, SearchParameters}
import com.xebialabs.xlplatform.xlrepository.tck.RepositoryFactory
import com.xebialabs.xlplatform.xlrepository.tck.cis._
import com.xebialabs.xlplatform.xlrepository.tck.util.RepositorySuiteBase

import scala.collection.JavaConverters._
import scala.util.{Random, Try}

trait PerformanceSuite extends RepositorySuiteBase { self: RepositoryFactory[_] =>
  private val gen = generator.genFor("tck-perf")

  val pw = new PrintWriter(new FileOutputStream(new File("results.csv" )))

  val numTasks = 3
  val numPhases = 3

  val repeats = 1000

  override def afterAll(): Unit = {
    pw.close
  }

  private def createTasks(phaseId: String, numTasks: Int): java.util.List[Task] = {
    val tasks = for (i <- 1 to numTasks) yield
      ciHelper.constructConfigurationItem(classOf[Task], s"""$phaseId/task$i""")
    tasks.toList.asJava
  }

  private def createPhases(releaseId: String, numPhases: Int): java.util.List[Phase] = {
    val phases = for (i <- 1 to numPhases) yield
      ciHelper.constructConfigurationItem(classOf[Phase], s"""$releaseId/phase$i""")
    phases.toList.asJava
  }

  private def createRelease(releaseId: String) = ciHelper.constructConfigurationItem(classOf[Release], releaseId)

  private def generateData(numReleases: Int): List[String] = {
    val releases = (1 to numReleases) map(_ => gen.nextId)
    for (releaseId <- releases) {
      val release = createRelease(releaseId)
      release.phase = createPhases(releaseId, numPhases)
      release.phase forEach(phase => phase.task = createTasks(phase.getId, numTasks))
      val tasks = release.phase.asScala.flatMap({p => p.task.asScala})

      addReferences(tasks.toList)

      repository.createCollection((List(release) ++ release.phase.asScala ++ tasks).asJava)
    }
    releases.toList
  }

  private def addReferences(tasks: List[Task]) = {
    val random = new Random
    val numTasks = tasks.size
    for (i <- 1 to numTasks / 4) {
      val referencer = tasks(random.nextInt(numTasks))
      val referenced1 = tasks(random.nextInt(numTasks))
      val referenced2 = tasks(random.nextInt(numTasks))
      referencer.dependencies = List(referenced1.getId, referenced2.getId).filterNot(_ == referencer.getId).asJava
    }
  }

  private def newApp: Release = {
    ciHelper.constructCi(Type.valueOf(classOf[Release]), gen.nextId)
  }

  // Runs a function block for a given number of times and returns the average runtime
  def time[R](func: => R, name: String, times: Int = repeats): Long = {
    val total = (1 to times).foldLeft(0L) {
      case (acc, _) => {
        val t0 = System.currentTimeMillis()
        func
        val t1 = System.currentTimeMillis()
        acc + (t1 - t0)
      }
    }

    val avg = total /  times
    pw.write(s"${name}, ${avg}ms\n")
    avg
  }


  describe("Performance Testing for repository API ") {

    it("should time if exists") {
      val releases: List[String] = generateData(100)
      time ({ repository.exists(releases.head) }, "if exists")
    }

    it("should time reading a release") {
      val releases: List[String] = generateData(100)
      time ({ repository.read[Release](releases.head) }, "reading a release")
    }

    it("should time reading a release with a shallow depth") {
      val releases: List[String] = generateData(100)
      time ({ repository.read[Release](releases.head, 1) }, "reading a release with a shallow depth")
    }

    it("should time reading using a cache") {
      val releases: List[String] = generateData(2)
      time ({
        repository.read[Release](releases.head, true)
        repository.read[Release](releases.head, true)
      }, "reading using a cache")
    }

    // TODO Relevant only after artifacts are implemented
//    it("should time reading a release with a workdir") {
//      val releases: List[String] = generateData(100)
//      val workDirFile = mk(folder("workDir"))
//      time ({ repository.read[Release](releases.head, new WorkDir(LocalFile.from(workDirFile)), false) },
//        "reading a release with a workdir")
//    }
    // TODO Relevant only after artifacts are implemented
//    it("should time reading a release with a workdir and depth") {
//      val releases: List[String] = generateData(100)
//      val workDirFile = mk(folder("workDir"))
//      time ({ repository.read[Release](releases.head, 1, new WorkDir(LocalFile.from(workDirFile)), false) },
//        "reading a release with a workdir and depth")
//    }

    it("should time reading a list of release") {
      val releases = generateData(100)
      time ({ repository.read[Release](releases.asJava, 1, false) }, "reading a list of release")
    }

    it("should time listing all releases with search parameters") {
      generateData(200)
      time ( { repository.listEntities(new SearchParameters) }, "listing all releases with search parameters")
    }

    it("should time creating a release") {
      generateData(100)

      time ({
        val release = ciHelper.constructConfigurationItem(classOf[Release], gen.nextId)
        repository.create(release)
      }, "creating a release")
    }

    it("should time updating a release") {
      val releases: List[String] = generateData(100)
      val updatedRelease = ciHelper.constructConfigurationItem(classOf[Release], Random.shuffle(releases).head)
      val phase1 = ciHelper.constructConfigurationItem(classOf[Phase], s"${releases.head}/phase1")
      time ( { repository.update(phase1, updatedRelease) }, "updating a release")
    }

    it("should time creating a release with the create or update call") {
      generateData(100)
      time ({
        val release = ciHelper.constructConfigurationItem(classOf[Release], gen.nextId)
        repository.createOrUpdate(release) }, "creating a release with the create or update call")
    }

    it("should time moving a release") {
      generateData(5)
      val sp = new SearchParameters().setType(Type.valueOf(classOf[Release]))
      time ({
        val releases = repository.list(sp).asScala.map(_.getId)
        repository.move(Random.shuffle(releases).head, gen.nextId)
      }, "moving a release")
    }

    it("should time renaming a release") {
      generateData(5)
      val sp = new SearchParameters().setType(Type.valueOf(classOf[Release]))
      time ({
        val releases = repository.list(sp).asScala.map(_.getId)
        repository.rename(Random.shuffle(releases).head, s"BrandNewId${Random.nextLong}")
      }, "renaming a release")
    }

    it("should time deleting a release") {
      val releases = generateData(50)
      val sp = new SearchParameters().setType(Type.valueOf(classOf[Release]))
      time ({
        // Also testing deletion of cis which were already deleted
        repository.delete(Random.shuffle(releases).head)
      }, "deleting a release")
    }

    it("should time executing a changeset") {
      generateData(100)
      time ({
        val set: ChangeSet = new ChangeSet()
        set.create(List(newApp, newApp, newApp).asJava)
        repository.execute(set) }, "executing a changeset")
    }

    /**
      * TODO: These methods haven't been implemented in the sql implementation yet
      **
      *it("should time copying a release") {
      *val release = generateData(100)
      *repository.copy(release.head, "Configuration/newReleaseId")
      * }
      **
      *it("should time checking referential integrity") {
      *generateData(100)
      *val set: ChangeSet = new ChangeSet()
      *set.create(List(newApp, newApp, newApp).asJava)
      *repository.checkReferentialIntegrity(set)
      * }
      **
      *it("should time returning query template factory") {
      *generateData(100)
      *repository.getQueryTemplateFactory
      *}
    */
  }

  describe("Performance testing of scenarios") {

    it("should time reading a release and then modifying it") {
      val releaseList: List[String] = generateData(50)
      time ({
        val release = repository.read[Release](releaseList.head)
        repository.update(release)
      }, "reading a release and then modifying it")
    }

    it("should time reading a release and then deleting a task") {
      val releaseList: List[String] = generateData(50)
      time({
        val release = repository.read[Release](releaseList.asJava.get(Random.nextInt(releaseList.size)))
        // in Try clause, in case the task was already deleted
        Try(release.phase.get(0).task.get(0))
          .map(task => repository.delete(task.getId))
      }, "reading a release and then deleting a task")
    }
  }

}
