package com.xebialabs.deployit.repository;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.EncryptedStringValue;
import com.xebialabs.deployit.core.StringValue;
import com.xebialabs.deployit.util.PasswordEncrypter;

import java.util.function.Function;

public class StringValueConverter {

    public static final String ENCRYPTED_PREFIX = "e{";
    public static final String PLAIN_PREFIX = "u{";
    public static final String SUFFIX = "}";

    private final PasswordEncrypter passwordEncrypter;

    public StringValueConverter(PasswordEncrypter passwordEncrypter) {
        this.passwordEncrypter = Checks.checkNotNull(passwordEncrypter, "passwordEncrypter");
    }

    public static final Function<StringValue, String> valueToString(PasswordEncrypter passwordEncrypter) {
        return valueToString(passwordEncrypter, false);
    }

    public static final Function<StringValue, String> valueToString(PasswordEncrypter passwordEncrypter, final boolean fallbackEncrypted) {
        final StringValueConverter converter = new StringValueConverter(passwordEncrypter);
        return input -> converter.convert(input, fallbackEncrypted);
    }

    public static final Function<String, StringValue> stringToValue(PasswordEncrypter passwordEncrypter) {
        final StringValueConverter converter = new StringValueConverter(passwordEncrypter);
        return input -> converter.convert(input);
    }

    /**
     * Convert a StringValue to its public representation. For an {@link com.xebialabs.deployit.core.EncryptedStringValue} this means
     * that the value will be encrypted. For a plaintext {@link com.xebialabs.deployit.core.StringValue} it adds a prefix/suffix to the string
     * if the boolean <i>keepPlainAsIs</i> is set to <code>true</code>, the plaintext is output as is.
     * @param v the {@link com.xebialabs.deployit.core.StringValue} to convert
     * @param keepPlainAsIs <code>true</code> to keep the plainText as is, <code>false</code> to add a special prefix/suffix to clearly demarcate the value.
     * @return The converted String.
     */
    public String convert(StringValue v, boolean keepPlainAsIs) {
        if (v instanceof EncryptedStringValue) {
            return ENCRYPTED_PREFIX + passwordEncrypter.ensureEncrypted(v.toString()) + SUFFIX;
        } else if (!keepPlainAsIs) {
            return PLAIN_PREFIX + v.toString() + SUFFIX;
        } else {
            return v.toString();
        }
    }

    String convert(StringValue v) {
        return convert(v, false);
    }

    public StringValue convert(String s) {
        return convert(s, false);
    }

    public StringValue convert(String s, boolean fallbackEncrypted) {
        if (s.startsWith(ENCRYPTED_PREFIX)) {
            String substring = s.substring(ENCRYPTED_PREFIX.length(), s.length() - SUFFIX.length());
            return new EncryptedStringValue(passwordEncrypter.ensureDecrypted(substring));
        } else if (s.startsWith(PLAIN_PREFIX)) {
            String substring = s.substring(PLAIN_PREFIX.length(), s.length() - SUFFIX.length());
            return new StringValue(substring);
        } else {
            if (fallbackEncrypted) {
                return new EncryptedStringValue(s);
            } else {
                // FALLBACK SCENARIO
                return new StringValue(s);
            }
        }
    }
}
