package com.xebialabs.deployit.security;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.jcr.*;

import org.apache.jackrabbit.api.JackrabbitSession;
import org.apache.jackrabbit.api.security.user.*;
import org.apache.jackrabbit.api.security.user.User;
import org.apache.jackrabbit.core.security.authentication.CryptedSimpleCredentials;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.base.Preconditions;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.security.authentication.AuthenticationFailureException;
import com.xebialabs.deployit.security.authentication.UserAlreadyExistsException;

@Component
public class JackrabbitUserService implements UserService {

    private final JcrTemplate jcrTemplate;

    @Autowired
    public JackrabbitUserService(JcrTemplate jcrTemplate) {
        this.jcrTemplate = jcrTemplate;
    }

    @Override
    public void create(final String username, final String password) {
        jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws RepositoryException {
                UserManager um = ((JackrabbitSession) session).getUserManager();

                try {
                    um.createUser(username, password);
                } catch (AuthorizableExistsException e) {
                    throw new UserAlreadyExistsException(username);
                }
                return null;
            }
        });
    }

    @Override
    public JcrUser read(final String username) {
        return jcrTemplate.execute(new JcrCallback<JcrUser>() {
            @Override
            public JcrUser doInJcr(Session session) throws RepositoryException {
                User u = getUser(username, session);
                return new JcrUser(u.getID(), u.isAdmin());
            }
        });

    }

    @Override
    public List<String> listUsernames() {
        return jcrTemplate.execute(new JcrCallback<List<String>>() {
            @Override
            public List<String> doInJcr(Session session) throws RepositoryException {
                UserManager um = ((JackrabbitSession) session).getUserManager();
                Iterator<Authorizable> authorizables = um.findAuthorizables(new Query() {
                    @Override
                    public <T> void build(QueryBuilder<T> builder) {
                        builder.setSelector(User.class);
                    }
                });
                List<String> names = new ArrayList<>();
                while (authorizables.hasNext()) {
                    String name = authorizables.next().getPrincipal().getName();
                    if ("anonymous".equals(name)) continue;
                    names.add(name);
                }
                return names;
            }
        });
    }

    @Override
    public void modifyPassword(final String username, final String newPassword) {
        jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws RepositoryException {
                User u = getUser(username, session);
                u.changePassword(newPassword);
                return null;
            }
        });

    }

    @Override
    public void modifyPassword(final String username, final String newPassword, final String oldPassword) {
        jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws RepositoryException {
                User u = getUser(username, session);
                u.changePassword(newPassword, oldPassword);
                return null;
            }
        });

    }

    @Override
    public void delete(final String username) {
        jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws RepositoryException {
                User u = getUser(username, session);
                u.remove();
                return null;
            }
        });
    }

    @Override
    public void authenticate(final String username, final String password) throws AuthenticationFailureException {
        jcrTemplate.execute(new JcrCallback<Object>() {
            @Override
            public Object doInJcr(Session session) throws RepositoryException {
                UserManager um = ((JackrabbitSession) session).getUserManager();
                User user = (User) um.getAuthorizable(username);
                if (user == null) {
                    throw new AuthenticationFailureException("Cannot authenticate [%s],", username);
                }
                Credentials credentials = user.getCredentials();
                validateCredentials(credentials, username, password);
                return null;
            }
        });
    }

    private User getUser(final String username, Session session) throws AccessDeniedException, UnsupportedRepositoryOperationException, RepositoryException {
        UserManager um = ((JackrabbitSession) session).getUserManager();

        Authorizable authorizable = um.getAuthorizable(username);
        if (authorizable == null) {
            throw new NotFoundException("No such user: " + username);
        }

        if (authorizable.isGroup()) {
            throw new IllegalStateException("Principal " + username + " is not a user");
        }

        User u = (User) authorizable;
        return u;
    }


    private void validateCredentials(Credentials credentials, String username, String password) {
        Preconditions.checkState(credentials instanceof CryptedSimpleCredentials, "Should have an instance of CryptedSimpleCredentials");
        try {
            if (!((CryptedSimpleCredentials) credentials).matches(new SimpleCredentials(username, password.toCharArray()))) {
                throw new AuthenticationFailureException("Wrong credentials supplied for user [%s]", username);
            }
        } catch (NoSuchAlgorithmException | UnsupportedEncodingException e) {
            throw new AuthenticationFailureException(e);
        }
    }

}
