package com.xebialabs.xlplatform.security.sql

import java.util

import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{PermissionChecker, Role}
import com.xebialabs.xlplatform.repository.sql.Database
import com.xebialabs.xlplatform.security.sql.db.Ids.isGlobalId
import com.xebialabs.xlplatform.security.sql.db.Tables
import grizzled.slf4j.Logging
import org.springframework.security.core.Authentication
import slick.jdbc.JdbcProfile

import scala.collection.JavaConverters._

class SqlPermissionChecker(securityDatabase: Database) extends PermissionChecker with Logging {

  import securityDatabase._

  val profile: JdbcProfile = config.databaseType.profile

  import profile.api._

  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, roles: util.List[Role]): Boolean =
    checkPermission(permissions, onConfigurationItem, roles, getAuthentication)

  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, roles: util.List[Role], auth: Authentication): Boolean =
    checkPermission(permissions.asScala, Option(onConfigurationItem), roles.asScala, auth)

  protected def checkPermission(permissions: Seq[Permission], onConfigurationItem: Option[String], roles: Seq[Role], auth: Authentication): Boolean = {
    val target = if (isGlobalId(onConfigurationItem)) "global" else onConfigurationItem.get
    logger.debug(s"Checking permissions [${permissions.mkString(", ")}] on [$target]")

    val allRoles = if (isGlobalId(onConfigurationItem)) roles else roles ++ getRoles(onConfigurationItem, roles, auth)

    val hasPermission = runAwait {
      Tables.roles
        .join(Tables.rolePermissions).on(_.id === _.roleId)
        .filter { case (role, rolePermissions) =>
          role.id.in(allRoles.map(_.getId)) &&
            rolePermissions.isOnConfigurationItem(onConfigurationItem) &&
            rolePermissions.permissionName.in(permissions.map(_.getPermissionName))
        }.exists.result
    }

    logger.debug(s"Permissions [${permissions.mkString(", ")}] granted on [$target]: $hasPermission")

    hasPermission
  }

  private def getRoles(onConfigurationItem: Option[String], roles: Seq[Role], auth: Authentication): Seq[Role] = {
    logger.debug(s"Reading local roles on CI [${onConfigurationItem.get}]")

    val localRoles = runAwait {
      Tables.roles
        .joinLeft(Tables.rolePrincipals).on { case (role, rolePrincipal) => role.id === rolePrincipal.roleId }
        .joinLeft(Tables.roleRoles).on { case ((role, _), roleRole) => role.id === roleRole.roleId }
        .filter { case ((role, rolePrincipals), roleRoles) =>
          role.isOnConfigurationItem(onConfigurationItem) &&
            (rolePrincipals.map(_.principalName).in(authenticationToPrincipals(auth).asScala) ||
              roleRoles.map(_.memberRoleId).in(roles.map(_.getId)))
        }.map(_._1._1).result
    }.map(role => new Role(role.id, role.name))

    logger.debug(s"Found roles [${localRoles.mkString(", ")}] on CI [${onConfigurationItem.get}]")
    localRoles
  }
}
