package com.xebialabs.xlplatform.support.rest

import com.xebialabs.deployit.core.api.dto.InternalReport
import com.xebialabs.xlplatform.support.api.SupportService
import com.xebialabs.xlplatform.utils.ResourceManagement._
import grizzled.slf4j.Logging
import org.apache.commons.lang3.exception.ExceptionUtils
import org.springframework.stereotype.Component

import java.io._
import java.nio.charset.{Charset, StandardCharsets}
import java.nio.file.Files
import java.util
import java.util.zip.{ZipEntry, ZipOutputStream}

@Component
class SupportServiceImpl extends SupportService with Logging {

  private val passwordFileExtentions = Seq(".xml", ".conf")
  private val passwordReplacement = "<line contained a password and was removed>"
  private val encryptedKeyReplacement = "<line contained a secret key and was removed>"

  override def prepareSupportZip(output: OutputStream,
                                 directoriesToZip: util.List[String],
                                 directoriesToList: util.List[String],
                                 reportsToZip: util.List[InternalReport]): Unit =
    using(new ZipOutputStream(asBuffered(output))) { zipOutputStream => {
      directoriesToZip.forEach(dir => addToZip(zipOutputStream, new File(dir), None))
      directoriesToList.forEach(dir => listToZip(zipOutputStream, new File(dir)))
      reportsToZip.forEach(reportPath => reportToZip(zipOutputStream, reportPath))
      zipOutputStream.flush()
    }}

  private def asBuffered(os: OutputStream): OutputStream =
    if (os.isInstanceOf[BufferedOutputStream]) os
    else new BufferedOutputStream(os)

  private def listToZip(zipOutputStream: ZipOutputStream, file: File): Unit = {
    if (file.isDirectory) {
      val listedDirectory = listDirectoryRecursively(file, new StringBuilder(30) , 0)
      withNewZipEntry(s"${file.getName}.txt", zipOutputStream,
        () => zipOutputStream.write(listedDirectory.toString().getBytes(Charset.forName("UTF-8"))))
    } else {
      logger.info(s"The file [${file.getCanonicalFile}] is not a directory and can't be listed.")
    }
  }

  private def listDirectoryRecursively(file: File, accumulator: StringBuilder, depth: Int): StringBuilder = {
    accumulator.append("  " * depth).append(file.getName).append("\n")
    file.listFiles().foreach( f => {
      if (f.isDirectory) {
        listDirectoryRecursively(f, accumulator, depth + 1)
      } else {
        accumulator.append("  " * depth).append("  - ").append(f.getName).append("\n")
      }
    })
    accumulator
  }

  private def addToZip(zipOutputStream: ZipOutputStream, fileToZip: File, maybeParentDirectoryName: Option[String]): Unit = {
    withValidFile(fileToZip) match {
      case Some(file) =>
        val zipEntryName = maybeParentDirectoryName.map(parent => s"$parent/${file.getName}").getOrElse(file.getName)

        file match {
          case f if f.isDirectory => f.listFiles.foreach(f => addToZip(zipOutputStream, f, Option(zipEntryName)))
          case f if shouldScrubPasswordsInFile(f.getName) => zipFileWithScrubPasswords(zipOutputStream, f, zipEntryName)
          case f => zipFile(zipOutputStream, f, zipEntryName)
        }
      case _ =>
    }
  }

  private def reportToZip(zipOutputStream: ZipOutputStream, internalReport: InternalReport): Unit = {
    try {
      val reportData = internalReport.getReportSupplier.get()
      withNewZipEntry(s"${internalReport.getReportName}.${internalReport.getFileType}", zipOutputStream,
        () => zipOutputStream.write(reportData.getBytes(StandardCharsets.UTF_8)))
    } catch {
      case e: Exception =>
        val exceptionData = s"${e.getMessage}${System.lineSeparator}${ExceptionUtils.getStackTrace(e)}"
        withNewZipEntry(s"${internalReport.getReportName}.err", zipOutputStream,
          () => zipOutputStream.write(exceptionData.getBytes(StandardCharsets.UTF_8)))
    }
  }

  private def zipFileWithScrubPasswords(zipOutputStream: ZipOutputStream, fileToZip: File, zipEntryName: String): Unit = {
    using(new BufferedReader(new FileReader(fileToZip))) { reader =>
      withNewZipEntry(zipEntryName, zipOutputStream, () => {
        while (reader.ready) {
          val original = reader.readLine
          val line = if (shouldScrubPasswordsInLine(original)) passwordReplacement else if (shouldScrubEncryptedKeyInLine(original)) encryptedKeyReplacement else original
          zipOutputStream.write((line + System.lineSeparator).getBytes)
        }
      })
    }
  }

  private def zipFile(zipOutputStream: ZipOutputStream, fileToZip: File, zipEntryName: String): Unit =
    withNewZipEntry(zipEntryName, zipOutputStream, () => Files.copy(fileToZip.toPath, zipOutputStream))

  private def withNewZipEntry(zipEntryName: String, zipOutputStream: ZipOutputStream, writer: () => Unit): Unit = {
    zipOutputStream.putNextEntry(new ZipEntry(zipEntryName))
    writer()
    zipOutputStream.closeEntry()
  }

  private def shouldScrubPasswordsInFile(fileName: String): Boolean =
    passwordFileExtentions.exists(ext => fileName.endsWith(ext))

  private def shouldScrubPasswordsInLine(line: String): Boolean = line.contains("password")

  private def shouldScrubEncryptedKeyInLine(line: String): Boolean = line.contains("encrypt.key")


  private def withValidFile(fileToZip: File): Option[File] = {
    fileToZip match {
      case file if file == null =>
        logger.info("The file or directory is invalid and cannot be processed")
        None
      case file if !file.exists() =>
        logger.info(s"The file or directory [$file] does not exist and will not be processed")
        None
      case file if file.getName.endsWith(".lck") =>
        logger.info(s"The file or directory [${file.getName}] is a service lock file and will not be processed")
        None
      case file => Option(file)
    }
  }
}
