package com.xebialabs.xlplatform.test.ci

import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.overthere.OverthereFile

import scala.collection.convert.wrapAsJava._
import scala.language.implicitConversions

object CiHelper {
  implicit def stringToType(t: String): Type = Type.valueOf(t)

  implicit def classToType[T <: ConfigurationItem](c: Class[T]): Type = Type.valueOf(c)

  case class ProvisioningInfo(templates: Set[Template], boundTemplates: Set[Template] = Set())

}

class CiHelper(repositoryService: RepositoryService) {

  import com.xebialabs.xlplatform.test.ci.CiHelper._

  def constructCi[T <: ConfigurationItem](t: Type, id: String): T = {
    DescriptorRegistry.getDescriptor(t).newInstance(id)
  }

  def createCi[T <: ConfigurationItem](t: Type, id: String): T = {
    val ci: T = constructCi(t, id)
    repositoryService.create(ci)
    repositoryService.read(ci.getId)
  }

  def constructConfigurationItem[T <: ConfigurationItem](clazz: Class[T], id: String): T = constructCi(clazz, id)

  def createConfigurationItem[T <: ConfigurationItem](clazz: Class[T], id: String): T = createCi(clazz, id)

  def createDirectory(id: String): Directory = createConfigurationItem(classOf[Directory], id)

  def constructArtifact[T <: Artifact](clazz: Class[T], id: String, file: OverthereFile): T = {
    val ci = constructConfigurationItem(clazz, id)
    ci.setFile(file)
    ci
  }

  def createArtifact[T <: Artifact](clazz: Class[T], id: String, file: OverthereFile): T = {
    val ci = constructArtifact(clazz, id, file)
    repositoryService.create(ci)
    repositoryService.read(ci.getId)
  }

  def createApplication(id: String): Application = createConfigurationItem(classOf[Application], "Applications/" + id)

  def constructPackage(id: String, app: Application, members: Deployable*): DeploymentPackage = {
    val pkg = constructConfigurationItem(classOf[DeploymentPackage], app.getId + "/" + id)
    pkg.setApplication(app)
    pkg.setDeployables(newHashSet(members.toSet))
    pkg
  }


  def constructProvisioningPackage(id: String, app: Application, members: Provisionable*): ProvisioningPackage = {
    constructProvisioningPackage(id, app, None, members: _*)
  }

  def constructProvisioningPackage(id: String, app: Application, provisioningInfo: Option[ProvisioningInfo], members: Provisionable*): ProvisioningPackage = {
    val pkg = constructConfigurationItem(classOf[ProvisioningPackage], app.getId + "/" + id)
    pkg.setApplication(app)
    pkg.setDeployables(newHashSet(members.toSet))
    provisioningInfo.foreach(p => {
      pkg.setTemplates(p.templates)
      pkg.setBoundTemplates(p.boundTemplates)
    })
    pkg
  }

  def createPackage(id: String, app: Application, members: Deployable*): DeploymentPackage = {
    val pkg = constructPackage(id, app, members: _*)
    repositoryService.create(pkg)
    repositoryService.read(pkg.getId)
  }

  def createComposite(id: String, app: Application, packages: Version*): CompositePackage = {
    val composite = constructConfigurationItem(classOf[CompositePackage], app.getId + "/" + id)
    composite.setApplication(app)
    composite.setProperty("packages", newArrayList(packages))
    repositoryService.create(composite)
    repositoryService.read(composite.getId)
  }

  def createContainer[C <: Container](id: String, clazz: Class[C]): C = createConfigurationItem(clazz, "Infrastructure/" + id)

  def createEnvironment(id: String, members: Container*): Environment = {
    val env = constructConfigurationItem(classOf[Environment], "Environments/" + id)
    env.setMembers(newHashSet(members.toSet))
    repositoryService.create(env)
    repositoryService.read(env.getId)
  }

  def constructDeployed[D <: Deployable, C <: Container, DC <: Deployed[D, C]](clazz: Class[DC], deployable: D, container: C): DC = {
    val deployed = constructConfigurationItem(clazz, container.getId + "/" + deployable.getId.split("/").last)
    deployed.setDeployable(deployable.asInstanceOf[D])
    deployed.setContainer(container)
    deployed
  }

  def constructDeployment(version: Version, env: Environment, deployeds: Deployed[_ <: Deployable, _ <: Container]*): DeployedApplication = {
    val deployment = constructConfigurationItem(classOf[DeployedApplication],
      env.getId + "/" + version.getApplication.getId.split("/").last)
    deployment.setVersion(version)
    deployment.setEnvironment(env)
    deployment.setDeployeds(newHashSet(deployeds.toSet))
    deployment
  }

  def createDeployment(version: Version, env: Environment,
                       deployeds: Deployed[_ <: Deployable, _ <: Container]*): DeployedApplication = {
    val d = constructDeployment(version, env, deployeds: _*)
    repositoryService.create(d)
    repositoryService.read(d.getId)
  }

  // Define several shortcuts for methods with variable arguments to make life easier for Java clients of this helper

  def createProvisioningPackage(id: String, app: Application): ProvisioningPackage = {
    createProvisioningPackage(id, app, None, Seq(): _*)
  }

  def createProvisioningPackage(id: String, app: Application, provisioningInfo: Option[ProvisioningInfo], members: Provisionable*): ProvisioningPackage = {
    val pkg = constructProvisioningPackage(id, app, provisioningInfo, members: _*)
    repositoryService.create(pkg)
    repositoryService.read(pkg.getId)
  }

  def createPackage(id: String, app: Application): DeploymentPackage = {
    createPackage(id, app, Seq(): _*)
  }

  def createPackage(id: String, app: Application, member: Deployable): DeploymentPackage = {
    createPackage(id, app, Seq(member): _*)
  }

  def createPackage(id: String, app: Application, member1: Deployable, member2: Deployable): DeploymentPackage = {
    createPackage(id, app, Seq(member1, member2): _*)
  }

  def createEnvironment(id: String): Environment = {
    createEnvironment(id, Seq(): _*)
  }

  def createEnvironment(id: String, member: Container): Environment = {
    createEnvironment(id, Seq(member): _*)
  }

  def createEnvironment(id: String, member1: Container, member2: Container): Environment = {
    createEnvironment(id, Seq(member1, member2): _*)
  }

  def constructDeployment(version: Version, env: Environment): DeployedApplication = {
    constructDeployment(version, env, Seq(): _*)
  }

  def constructDeployment(version: Version, env: Environment,
                          deployed: Deployed[_ <: Deployable, _ <: Container]): DeployedApplication = {
    constructDeployment(version, env, Seq(deployed): _*)
  }

  def createDeployment(version: Version, env: Environment): DeployedApplication = {
    createDeployment(version, env, Seq(): _*)
  }

  // Make sure that converted maps are serializable
  private def newHashSet[T](s: Set[T]) = new java.util.HashSet[T](scala.collection.JavaConversions.setAsJavaSet(s))

  private def newArrayList[T](s: Seq[T]) = new java.util.ArrayList[T](scala.collection.JavaConversions.seqAsJavaList(s))

}
