package com.xebialabs.xlplatform.test.modeshape;

import java.io.InputStream;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import org.modeshape.common.collection.Problem;
import org.modeshape.common.collection.Problems;
import org.modeshape.jcr.JcrRepository;
import org.modeshape.jcr.ModeShapeEngine;
import org.modeshape.jcr.RepositoryConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.xlplatform.repository.RepositoryConfigurationSource;
import com.xebialabs.xlplatform.repository.XlJcrRepositoryInitializer;
import com.xebialabs.xlplatform.repository.XlRepositoryConfig;
import com.xebialabs.xlplatform.test.jcr.RepositoryProvider;

import static com.xebialabs.xlplatform.test.modeshape.TestModeshapeRepositoryConfig.CONFIGURATION_LOCATION;

public class TestModeshapeRepositoryBooter implements RepositoryProvider {
    private static final Logger logger = LoggerFactory.getLogger(TestModeshapeRepositoryBooter.class);

    private JcrRepository repository;

    private ModeShapeEngine engine;

    public Repository boot(XlRepositoryConfig config) throws RepositoryException {
        logger.info("Booting test modeshape repository with config {}", config.repositoryName());
        engine = new ModeShapeEngine();
        engine.start();
        String configLocation = config.repositoryConfig().getString(CONFIGURATION_LOCATION);
        RepositoryConfiguration modeShapeConfig = readConfig(new RepositoryConfigurationSource(configLocation));
        repository = engine.deploy(modeShapeConfig);
        new XlJcrRepositoryInitializer(config).init(repository);
        logger.debug("Successfully initialized the repository");
        return repository;
    }

    private RepositoryConfiguration readConfig(RepositoryConfigurationSource source) throws RepositoryException {
        try (InputStream is = source.getInputStream()) {
            RepositoryConfiguration config = RepositoryConfiguration.read(is, source.getName());
            Problems problems = config.validate();
            if (!problems.isEmpty()) {
                for (Problem problem : problems) {
                    logger.error(problem.getMessageString());
                }
                throw new RepositoryException("Could not boot due to problems");
            }
            return config;
        } catch (RepositoryException re) {
            throw re;
        } catch (Exception e) {
            throw new RepositoryException(e);
        }
    }

    public void shutdown() {
        logger.info("Shutting down modeshape repository");
        Future<Boolean> shutdown = engine.shutdown();
        repository = null;
        engine = null;
        try {
            if (!shutdown.get()) {
                throw new IllegalStateException("Did not shutdown...");
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Failed to shutdown", e);
        } catch (ExecutionException e) {
            throw new IllegalStateException("Failed to shutdown", e);
        }
    }

    public JcrRepository getRepository() {
        return repository;
    }
}
