package com.xebialabs.xlplatform.ui

import akka.actor.ActorSystem
import akka.http.scaladsl.marshalling._
import akka.http.scaladsl.model._
import akka.http.scaladsl.model.ContentTypes.{`application/json`, `text/xml(UTF-8)`}
import akka.http.scaladsl.server.Route
import akka.util.Timeout
import com.xebialabs.xlplatform.endpoints.{AuthenticatedData, ExtensionRoutes}
import grizzled.slf4j.Logging

import scala.concurrent.Future
import scala.concurrent.duration._
import scala.language.postfixOps
import scala.util.{Failure, Success}

class MetadataExtensionRoute extends ExtensionRoutes with MenuXmlProtocol with MenuJsonProtocol with Logging {
  private implicit val timeout = Timeout(2 seconds)

  def route(system: ActorSystem): AuthenticatedData => Route = (_: AuthenticatedData) => {
    lazy val menus = Future {
      UiMenus(Settings.XlDeploy.UiExtension.file(system.settings.config))
    } (system.dispatcher)

    pathPrefix("metadata") {
      (get|post) {
        onComplete(menus) {
          case Success(menus) =>
            logger.info(s"Found UI extensions: $menus")
            complete(menus)
          case Failure(e: Throwable) =>
            logger.error(s"UI extensions could not be loaded", e)
            complete(StatusCodes.InternalServerError -> e.getMessage)
        }
      }
    }
  }

  private val jsonMenuListToHttp: Marshaller[List[IMenu], HttpEntity.Strict] = Marshaller.withFixedContentType(`application/json`) {
    (obj: List[IMenu]) => HttpEntity(`application/json`, menuListFormat.write(obj.toVector).compactPrint)
  }

  private val textXmlMenuListToHttp: Marshaller[List[IMenu], HttpEntity.Strict] = Marshaller.withFixedContentType(`text/xml(UTF-8)`) {
    (obj: List[IMenu]) => HttpEntity(`text/xml(UTF-8)`, iMenuListToXml(obj).toString)
  }

  private val applicationXmlMenuListToHttp: Marshaller[List[IMenu], HttpEntity.Strict] = Marshaller.withFixedContentType(applicationXml) {
    (obj: List[IMenu]) => HttpEntity(applicationXml, iMenuListToXml(obj).toString)
  }

  implicit val combinedMarshallers: ToResponseMarshaller[List[IMenu]] = Marshaller.oneOf(
    `application/json`, applicationXml, `text/xml(UTF-8)`
  ) {
    case ContentType(MediaTypes.`application/xml`, Some(HttpCharsets.`UTF-8`)) => applicationXmlMenuListToHttp
    case `text/xml(UTF-8)` => textXmlMenuListToHttp
    case _ => jsonMenuListToHttp
  }
}