package com.xebialabs.xlplatform.ui

import akka.http.scaladsl.marshallers.xml.ScalaXmlSupport
import akka.http.scaladsl.marshalling.Marshaller
import akka.http.scaladsl.model.ContentTypes.`text/xml(UTF-8)`
import akka.http.scaladsl.model.{ContentType, HttpCharsets, MediaTypes}

import scala.language.implicitConversions
import scala.xml.{Elem, NodeSeq, Text}

trait UIPluginExtensionsXmlProtocol extends ScalaXmlSupport {
  implicit def optStringToOptText(opt: Option[String]): Option[Text] = opt.map(Text(_))

  def menuItemToXml(m: MenuItem): Elem =
    <menu-item icon={m.icon} label={m.label} uri={m.uri} weight={m.weight.toString} path-suffix={m.pathSuffix}>
    {for ((k, v) <- m.properties) yield <property name={k} value={v}/>}
  </menu-item>

  def menuSeparatorToXml(m: MenuSeparator): Elem = <menu-separator weight={m.weight.toString}/>

  def libraryToXml(l: ExtensionLibrary): Elem = <library name={l.library}/>

  def resourceToXml(j: JSResource): Elem = <resource path={j.path}/>

  def permissionToXml(name: String): Elem = <permission>{name}</permission>

  val menuToXml: Menu => NodeSeq = (m: Menu) =>
    <menu icon={m.icon} id={m.id} label={m.label} uri={m.uri} weight={m.weight.toString}>
    {
      m.items.map {
        case menu: Menu => menuToXml(menu)
        case menuItem: MenuItem => menuItemToXml(menuItem)
        case menuSeparator: MenuSeparator => menuSeparatorToXml(menuSeparator)
      }
    }
    {
      m.resources.map { resource: JSResource =>
        resourceToXml(resource)
      }
    }
    {
      m.permissions.map { permission: String =>
        permissionToXml(permission)
      }
    }
  </menu>

  val iMenuToXml: IMenu => NodeSeq = {
    case menu: Menu => menuToXml(menu)
    case menuItem: MenuItem => menuItemToXml(menuItem)
    case library: ExtensionLibrary => libraryToXml(library)
    case _ => throw new IllegalArgumentException("Only Menu, MenuItem or Library can be printed in the root element of XML file")
  }

  val applicationXml: ContentType.WithCharset = ContentType(MediaTypes.`application/xml`, HttpCharsets.`UTF-8`)

  val iMenuListToXml: List[IMenu] => NodeSeq = hs => <list>
    {hs.map(iMenuToXml)}
  </list>

  implicit val xmlMenuMarshaller: Marshaller[IMenu, NodeSeq] = Marshaller.withFixedContentType(`text/xml(UTF-8)`)(iMenuToXml)

  implicit val xmlMenuItemListMarshaller: Marshaller[List[IMenu], NodeSeq] =
    Marshaller.withFixedContentType(`text/xml(UTF-8)`)(iMenuListToXml)
}
