package com.xebialabs.xlplatform.ui

import java.net.URL

import scala.xml.{Elem, Node, NodeSeq}
import com.xebialabs.xlplatform.utils.ResourceUtils

sealed trait IMenu

sealed trait OrderedMenu extends Ordered[OrderedMenu] with IMenu {
  def weight: Int

  def label: String

  override def compare(that: OrderedMenu): Int =
    if (this.weight == that.weight) {
      this.label.compareTo(that.label)
    } else {
      this.weight - that.weight
    }
}

case class JSResource(path: String) extends OrderedMenu with IMenu {
  def label: String = path

  override def weight: Int = 1
}

case class MenuSeparator(weight: Int) extends OrderedMenu {
  def label = "--------"
}

case class ExtensionLibrary(library: String) extends IMenu

case class MenuItem(weight: Int, label: String, uri: String, pathSuffix: Option[String], properties: Map[String, String], icon: Option[String])
    extends OrderedMenu

case class Menu(id: String,
                weight: Int,
                label: String,
                uri: Option[String],
                items: List[OrderedMenu],
                resources: List[JSResource],
                permissions: List[String],
                icon: Option[String])
    extends OrderedMenu

object UiExtensions {
  import ResourceUtils._

  def apply(filePattern: String): List[IMenu] = {
    val uiPluginElements: Map[URL, Elem] = loadXmlResources(filePattern)

    def applyReferences(menus: List[OrderedMenu], elems: NodeSeq): List[Menu] = {
      val lookup: Map[String, Menu] = menus
        .collect({
          case m: Menu => (m.id, m)
        }).toMap

      var notFoundMenuIds = List[String]()

      val resultedMenus = (elems \ "menu-ref")
        .foldLeft(lookup)({
          case (l, n) =>
            val menuId: String = (n \ "@ref").text
            if (l.contains(menuId)) {
              val appendedMenu: Menu = appendItems(l(menuId), n)
              val appendedWithRefs = appendedMenu.copy(
                items = (appendedMenu.items.filterNot(_.isInstanceOf[Menu]) ++ applyReferences(appendedMenu.items, n)).sorted
              )
              l + ((menuId, appendedWithRefs))
            } else {
              notFoundMenuIds = menuId :: notFoundMenuIds
              l
            }
        }).values.toList

      if (notFoundMenuIds.nonEmpty) {
        throw new NotFoundException(s"Menus with the next IDs are not found: ${notFoundMenuIds mkString ", "}")
      }

      resultedMenus
    }

    def getResource(e: Node): JSResource = JSResource((e \ "@path").text)

    def getPermission(e: Node): String = e.text

    def getMenuItem(e: Node): MenuItem =
      MenuItem(
        (e \ "@weight").text.toInt,
        (e \ "@label").text,
        (e \ "@uri").text,
        (e \ "@path-suffix").headOption.map(_.text),
        (e \ "property").map(n => (n \ "@name").text -> (n \ "@value").text).toMap,
        (e \ "@icon").headOption.map(_.text)
      )

    def getMenu(e: Node): Menu =
      appendItems(Menu((e \ "@id").text,
                       (e \ "@weight").text.toInt,
                       (e \ "@label").text,
                       (e \ "@uri").headOption.map(_.text),
                       List(),
                       List(),
                       List(),
                       (e \ "@icon").headOption.map(_.text)),
                  e)

    def getMenuSeparator(e: Node): MenuSeparator = MenuSeparator((e \ "@weight").text.toInt)

    def appendItems(m: Menu, e: Node): Menu =
      m.copy(
        items = (m.items ++ e.child.collect({
          case node: Elem if node.label == "menu-item" => getMenuItem(node)
          case node: Elem if node.label == "menu" => getMenu(node)
          case node: Elem if node.label == "menu-separator" => getMenuSeparator(node)
        })).sorted,
        resources = m.resources ++ e.child.collect({
          case node: Elem if node.label == "resource" => getResource(node)
        }),
        permissions = m.permissions ++ e.child.collect({
          case node: Elem if node.label == "permission" => getPermission(node)
        })
      )

    def checkMenuIdsOnDuplication(uiPluginElements: Map[URL, Elem]): Unit = {
      val allLevelMenuIds: Map[URL, Seq[String]] = uiPluginElements.view.mapValues(e => (e \\ "menu").map(p => (p \ "@id").text)).toMap

      def findDuplicates[A](xs: Seq[A]): Seq[A] = xs.distinct.filter(x => xs.count(_ == x) > 1)

      def findURLsById(id: String): Seq[URL] = allLevelMenuIds.toSeq.filter(_._2.contains(id)).map(_._1)

      val duplicates: Seq[(String, Seq[URL])] = findDuplicates(allLevelMenuIds.values.flatten.toSeq).map { id: String =>
        (id, findURLsById(id))
      }

      if (duplicates.nonEmpty)
        throw new DuplicationException(duplicates)
    }

    checkMenuIdsOnDuplication(uiPluginElements)
    val menus: List[OrderedMenu] = uiPluginElements.values.toList.flatMap(e => e \ "menu").map(getMenu)

    def getLibrary(e: Node): ExtensionLibrary = ExtensionLibrary((e \ "@name").text)

    applyReferences(menus, uiPluginElements.values.flatten.toSeq) ++ uiPluginElements.values.toList
      .flatMap(e => e \ "library").map(getLibrary)
  }
}
