package com.xebialabs.xlplatform.ui

import org.apache.pekko.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import spray.json._

trait UIPluginExtensionsJsonProtocol extends SprayJsonSupport with DefaultJsonProtocol {
  implicit val iMenuFormat: JsonFormat[IMenu] = lazyFormat(new RootJsonFormat[IMenu] {
    override def write(obj: IMenu): JsValue = obj match {
      case menu: Menu => menu.toJson
      case menuItem: MenuItem => menuItem.toJson
      case menuSeparator: MenuSeparator => menuSeparator.toJson
      case library: ExtensionLibrary => library.toJson
    }

    override def read(json: JsValue): IMenu = json match {
      case menu: JsObject if menu.fields.isDefinedAt("id") => menu.convertTo[Menu]
      case menuItem: JsObject if menuItem.fields.isDefinedAt("label") => menuItem.convertTo[MenuItem]
      case menuSeparator: JsObject if menuSeparator.fields.isDefinedAt("weight") => menuSeparator.convertTo[MenuSeparator]
      case library: JsObject if library.fields.isDefinedAt("library") => library.convertTo[ExtensionLibrary]
    }
  })

  implicit val menuItemFormat: JsonFormat[MenuItem] = lazyFormat(jsonFormat6(MenuItem))
  implicit val menuSeparatorFormat: JsonFormat[MenuSeparator] = jsonFormat2(MenuSeparator)
  implicit val menuFormat: JsonFormat[Menu] = lazyFormat(jsonFormat8(Menu))
  implicit val resourceFormat: JsonFormat[JSResource] = lazyFormat(jsonFormat1(JSResource))
  implicit val libraryFormat: RootJsonFormat[ExtensionLibrary] = jsonFormat1(ExtensionLibrary)

  implicit object MenuSeparatorFormat extends JsonFormat[MenuSeparator] {
    override def read(json: JsValue): MenuSeparator = {
      val fields = json.asJsObject("MenuSeparator object expected").fields
      MenuSeparator(
        fields("weight").convertTo[Int],
        fields.get("label").map(_.convertTo[String]).getOrElse(UiExtensions.MENU_SEPARATOR_LABEL)
      )
    }

    override def write(menuSeparator: MenuSeparator): JsValue = {

      if (menuSeparator.label == UiExtensions.MENU_SEPARATOR_LABEL) {
        JsObject(
          "weight" -> menuSeparator.weight.toJson,
          "separator" -> true.toJson,
        )
      } else {
        JsObject(
          "weight" -> menuSeparator.weight.toJson,
          "label" -> menuSeparator.label.toJson,
          "separator" -> true.toJson,
        )
      }
    }
  }

  implicit object JsResourceFormat extends JsonFormat[JSResource] {
    override def read(json: JsValue): JSResource = {
      val fields = json.asJsObject("JSResource object expected").fields
      JSResource(
        fields("path").toString()
      )
    }

    override def write(resource: JSResource): JsValue = JsObject(
      "path" -> resource.path.toJson
    )
  }

  implicit val menuListFormat: JsonFormat[Vector[IMenu]] = lazyFormat(new RootJsonFormat[Vector[IMenu]] {
    override def read(json: JsValue): Vector[IMenu] = json match {
      case JsArray(elems) => elems.map(_.convertTo[IMenu])
      case _ => throw new Exception(s"JSON value of type ${json.getClass.getName} cannot be turned into a List[Menu]")
    }

    override def write(obj: Vector[IMenu]): JsValue = JsArray(obj.map(_.toJson))
  })

  implicit val resourceListFormat: JsonFormat[Vector[JSResource]] = lazyFormat(new RootJsonFormat[Vector[JSResource]] {
    override def read(json: JsValue): Vector[JSResource] = json match {
      case JsArray(elems) => elems.map(_.convertTo[JSResource])
      case _ => throw new Exception(s"JSON value of type ${json.getClass.getName} cannot be turned into a List[JSResource]")
    }

    override def write(obj: Vector[JSResource]): JsValue = JsArray(obj.map(_.toJson))
  })

  implicit val orderedMenuFormat: JsonFormat[OrderedMenu] = lazyFormat(new RootJsonFormat[OrderedMenu] {
    override def write(obj: OrderedMenu): JsValue = obj match {
      case menu: Menu => menu.toJson
      case menuItem: MenuItem => menuItem.toJson
      case menuSeparator: MenuSeparator => menuSeparator.toJson
      case resource: JSResource => resource.toJson
    }

    override def read(json: JsValue): OrderedMenu = json match {
      case menu: JsObject if menu.fields.isDefinedAt("id") => menu.convertTo[Menu]
      case menuItem: JsObject if menuItem.fields.isDefinedAt("label") => menuItem.convertTo[MenuItem]
      case menuSeparator: JsObject => menuSeparator.convertTo[MenuSeparator]
    }
  })

}
