package com.xebialabs.xlplatform.watcher

import akka.actor._
import com.xebialabs.xlplatform.settings.XlPlatformSettings
import com.xebialabs.xlplatform.watcher.FileWatchActor.Messages._

import java.nio.file.Path
import scala.concurrent.ExecutionContextExecutor


object FileWatchActor {
  def props: Props = Props(new FileWatchActor)

  object Messages {

    case class Watch(path: Path)

    case class StartWatching(path: Path)

    case class WatchStopped(path: Path)

    case class Poll()

    sealed trait SystemEvent

    case class Created(path: Path) extends SystemEvent

    case class Modified(path: Path) extends SystemEvent

    case class Deleted(path: Path) extends SystemEvent

  }

}

class FileWatchActor extends Actor with ActorLogging with Stash {
  private var subscriptions = List[Subscription]()

  override def preStart(): Unit = {
    if (XlPlatformSettings(context).FileWatch.pollingEnabled) {
      implicit val dispatcher: ExecutionContextExecutor = context.system.dispatcher
      log.info(s"File changes polling is enabled")
      val interval = XlPlatformSettings(context).FileWatch.interval
      context.system.scheduler.scheduleWithFixedDelay(interval, interval, self, Poll())
    } else {
      log.warning("File changes polling is disabled")
    }
  }

  override def postStop(): Unit = {
    subscriptions.foreach(_.stopPolling())
    subscriptions = Nil
  }

  override def receive: Receive = {
    case Watch(path) => startWatch(path)
    case Poll() => pollFileEvents()
  }

  private def startWatch(path: Path): Unit = {
    log.debug(s"Watching changes on path ${path}")
    val subscription = new Subscription(sender(), path)
    subscriptions = subscription +: subscriptions
    sender() ! StartWatching(path)
  }

  private def pollFileEvents(): Unit = {
    subscriptions.foreach(_.notifyIfFileModified())
  }
}

private[watcher] class Subscription(notifyActor: ActorRef, fileToWatch: Path) extends FileWatch(fileToWatch.toFile) {
  def notifyIfFileModified(): Unit = doWhenModified {
    case StateChange.Created => notifyActor ! Created(fileToWatch)
    case StateChange.Modified => notifyActor ! Modified(fileToWatch)
    case StateChange.Deleted => notifyActor ! Deleted(fileToWatch)
  }

  def stopPolling(): Unit = notifyActor ! WatchStopped(fileToWatch)
}
