package com.xebialabs.deployit.maven.converter;

import java.util.List;
import java.util.Map;

import org.codehaus.plexus.component.configurator.ComponentConfigurationException;
import org.codehaus.plexus.configuration.PlexusConfiguration;
import org.codehaus.plexus.configuration.PlexusConfigurationException;

import com.xebialabs.deployit.maven.MavenDeployable;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.maven.MavenDeployable.CiRefProperty;

public class MavenDeployableConverter {

    public static final String DEPLOYABLE = "deployable";

    public static MavenDeployable convertToDeployable(PlexusConfiguration configuration) throws ComponentConfigurationException {
        try {
            return doConvertToDeployable(configuration);
        } catch (Exception e) {
            throw new ComponentConfigurationException(e);
        }
    }

    private static MavenDeployable doConvertToDeployable(PlexusConfiguration configuration) throws PlexusConfigurationException {
        if (configuration.getName().equals(DEPLOYABLE)) {
            return doLegacyFormatConversion(configuration);
        } else {
            return doXmlManifestFormatConversion(configuration);
        }
    }

    private static MavenDeployable doXmlManifestFormatConversion(PlexusConfiguration configuration) throws PlexusConfigurationException {
        MavenDeployable deployable = new MavenDeployable();
        deployable.setType(configuration.getName());
        for (String name : configuration.getAttributeNames()) {
            deployable.addValue(name, configuration.getAttribute(name));
        }
        for (PlexusConfiguration propertyConfiguration : configuration.getChildren()) {
            String name = propertyConfiguration.getName();
            if (propertyConfiguration.getAttribute("ref") != null) {
                //ci reference
                deployable.addValue(name, new CiRefProperty(propertyConfiguration.getAttribute("ref")));
            } else if (propertyConfiguration.getChild("ci", false) != null) {
                //set or list of ci references
                deployable.addValue(name, convertCollectionOfCiReferences(propertyConfiguration.getChildren()));
            } else if (propertyConfiguration.getChild("value", false) != null) {
                //set of list of strings
                deployable.addValue(name, convertCollectionOfString(propertyConfiguration.getChildren()));
            } else if (propertyConfiguration.getChild("entry", false) != null) {
                //map of string string
                deployable.addValue(name, convertMapOfStringString(propertyConfiguration.getChildren()));
            } else if ("tags".equals(name)) {
                deployable.addValue(name, convertCollectionOfString(propertyConfiguration.getChildren()));
            } else if (propertyConfiguration.getChildCount() > 0) {
                //embedded
                deployable.addValue(name, convertEmbeddeds(propertyConfiguration.getChildren()));
            } else {
                //string, int, boolean or enum
                deployable.addValue(name, propertyConfiguration.getValue());
            }
        }

        return deployable;
    }

    private static MavenDeployable doLegacyFormatConversion(PlexusConfiguration configuration) throws PlexusConfigurationException {
        MavenDeployable deployable = new MavenDeployable();
        for (PlexusConfiguration propertyConfiguration : configuration.getChildren()) {
            String name = propertyConfiguration.getName();
            if (propertyConfiguration.getChild("list", false) != null) {
                PlexusConfiguration[] values = propertyConfiguration.getChild("list").getChildren();
                if (values.length > 0) {
                    if (values[0].getName().equals("value")) deployable.addValue(name, convertLegacyCollectionOfString("value", values));
                    else deployable.addValue(name, convertCollectionOfCiReferences(values));
                }
            } else if (propertyConfiguration.getChild("set", false) != null) {
                PlexusConfiguration[] values = propertyConfiguration.getChild("set").getChildren();
                if (values.length > 0) {
                    if (values[0].getName().equals("value")) deployable.addValue(name, convertLegacyCollectionOfString("value", values));
                    else deployable.addValue(name, convertCollectionOfCiReferences(values));
                }
            } else if (propertyConfiguration.getChild("map", false) != null) {
                PlexusConfiguration[] entries = propertyConfiguration.getChild("map").getChildren();
                deployable.addValue(name, convertLegacyMapOfStringString(entries));
            } else if (propertyConfiguration.getName().equals("tags")) {
                // here the property is <tags> and the above check just sees if there are any. If so, we should get
                // all children, which are each formed like <tag>blah</tag>
                PlexusConfiguration[] entries = propertyConfiguration.getChildren();
                deployable.addValue(name, convertLegacyCollectionOfString("tag", entries));
            } else {
                String c = propertyConfiguration.getValue();
                deployable.addValue(name, c);
            }
        }
        return deployable;
    }

    private static List<CiRefProperty> convertCollectionOfCiReferences(PlexusConfiguration[] ciRefConfig) throws PlexusConfigurationException {
        List<CiRefProperty> refs = newArrayList();
        for (PlexusConfiguration configuration : ciRefConfig) {
            if (configuration.getName().equals("ci")) {
                refs.add(new CiRefProperty(configuration.getAttribute("ref")));
            }
        }
        return refs;
    }

    private static List<String> convertCollectionOfString(PlexusConfiguration[] valueConfigs) throws PlexusConfigurationException {
        List<String> values = newArrayList();
        for (PlexusConfiguration configuration : valueConfigs) {
            if (configuration.getName().equals("value")) {
                values.add(configuration.getValue());
            }
        }
        return values;
    }

    private static Map<String, String> convertMapOfStringString(PlexusConfiguration[] valueConfigs) throws PlexusConfigurationException {
        Map<String, String> values = newHashMap();
        for (PlexusConfiguration configuration : valueConfigs) {
            if (configuration.getName().equals("entry")) {
                values.put(configuration.getAttribute("key"), configuration.getValue());
            }
        }
        return values;
    }

    private static List<MavenDeployable> convertEmbeddeds(PlexusConfiguration[] valueConfigs) throws PlexusConfigurationException {
        List<MavenDeployable> values = newArrayList();
        for (PlexusConfiguration configuration : valueConfigs) {
            values.add(doConvertToDeployable(configuration));
        }
        return values;
    }

    private static List<String> convertLegacyCollectionOfString(String xmlElementName, PlexusConfiguration[] valueConfigs) throws PlexusConfigurationException {
        List<String> values = newArrayList();
        for (PlexusConfiguration configuration : valueConfigs) {
            if (configuration.getName().equals(xmlElementName)) {
                values.add(configuration.getValue());
            }
        }
        return values;
    }

    private static Map<String, String> convertLegacyMapOfStringString(PlexusConfiguration[] valueConfigs) throws PlexusConfigurationException {
        Map<String, String> values = newHashMap();
        for (PlexusConfiguration configuration : valueConfigs) {
            values.put(configuration.getName(), configuration.getValue());
        }
        return values;
    }
}
