class XlrInputWithVariables {
    constructor(path, inputEditorGetter) {
        this.path = path;
        this.inputEditorGetter = inputEditorGetter;
        this.variablePicker = new XlWidgetAutocomplete(this.path);
        this.modeToggle = element(By.$(`${this.path} .variable-toggle .button`));
    }

    editor() {
        return this.inputEditorGetter();
    }

    variableSelector() {
        return this.variablePicker;
    }

    toggleMode() {
        this.modeToggle.click();
        return this;
    }

    expectEditMode() {
        expect(element(By.$(`${this.path} .editor`))).toBeDisplayed();
        return this;
    }

    expectVariableMode() {
        expect(element(By.$(`${this.path} .variable-dropdown`))).toBeDisplayed();
        return this;
    }

    expectToggleVisible(visible = true) {
        expect(this.modeToggle).toBePresent(visible);
        return this;
    }
}


class XlrInlinePasswordWithVariables extends XlrInputWithVariables {
    constructor(path) {
        super(path, function () {
            return new InlineEditor(`${path} .field`);
        });
        this.path = path;
    }
}

class XlrListStringWithVariables extends XlrInputWithVariables {
    constructor(path) {
        super(path, function () {
            return new DipListString(`${path} .field`);
        });
        this.path = path;
    }
}

class XlrSetStringWithVariables extends XlrInputWithVariables {
    constructor(path) {
        super(path, function () {
            return new DipSetString(`${path} .field`);
        });
        this.path = path;
    }
}

class XlrMapStringStringWithVariables extends XlrInputWithVariables {
    constructor(path) {
        super(path, function () {
            return new DipMapStringString(`${path} .field`);
        });
        this.path = path;
    }
}

class XlrIntegerWithVariables extends XlrInputWithVariables {
    constructor(path) {
        super(path, function () {
            return new XlDipInteger(`${path} .field`);
        });
        this.path = path;
    }
}

class XlrBooleanWithVariables extends XlrInputWithVariables {
    constructor(path) {
        super(path, function () {
            return new XlDipCheckbox(`${path} .field`);
        });
        this.path = path;
    }
}

class XlrDependencyWithVariables extends XlrInputWithVariables {
    constructor(path) {
        super(path, function () {
            return new DependencyEditor(`${path} .editor`);
        });
        this.path = path;
    }

    expectSaveButtonDisabled() {
        expect(element(By.$(`${this.path} .button.save`)).isEnabled()).toBe(false);
        return this;
    }

    expectSaveButtonEnabled() {
        expect(element(By.$(`${this.path} .button.save`)).isEnabled()).toBe(true);
        return this;
    }

    save() {
        element(By.$(`${this.path} .button.save`)).click();
        return this;
    }

    cancel() {
        element(By.$(`${this.path} .button.cancel`)).click();
        return this;
    }

    selectManualMode() {
        this.selectMode("Release, phase, or task");
        return this;
    }

    selectVariableMode() {
        this.selectMode("Variable");
        return this;
    }

    selectMode(mode) {
        return element(By.$(`${this.path} button:contains('${mode}')`)).click();
    }

    getLabel() {
        return element(By.$(`${this.path} .dependency-objects`)).getText();
    }

    getRelease() {
        return this._getValue('release-dependency');
    }

    setRelease(releaseTitle) {
        return this._setValue('release-dependency', releaseTitle);
    }
    clearRelease() {
        return this._clear('release-dependency');
    }

    getPhase() {
        return this._getValue('release-dependency');
    }

    setPhase(phaseTitle) {
        return this._setValue('phase-dependency', phaseTitle);
    }

    clearPhase() {
        return this._clear('phase-dependency');
    }

    expectPhase(present) {
        return expect(element.all(by.$(`${this.path} .dependency-objects .phase:visible`)).count()).toBe((present ? 1 : 0));
    }

    getTask() {
        return this._getValue('task-dependency');
    }

    setTask(taskTitle) {
        return this._setValue('task-dependency', taskTitle);
    }

    clearTask() {
        return this._clear('task-dependency');
    }

    expectTask(present) {
        return expect(element.all(by.$(`${this.path} .dependency-objects .task:visible`)).count()).toBe((present ? 1 : 0));
    }

    expectUnresolved() {
        return expect(element.all(by.$(`${this.path} .line .resolved`)).count()).toBe(0);
    }

    expectVariableModeEnabled(enabled) {
        expect(element(by.$(`${this.path} .variable-dependency-radio`)).isEnabled()).toBe(enabled);
    }

    _getValue(field) {
        const autocomplete = this._getAutocomplete(field);
        return autocomplete.getValue();
    }

    _setValue(field, value) {
        const autocomplete = this._getAutocomplete(field);
        autocomplete.focus().select(value, true);
        return this;
    }

    _clear(field) {
        const autocomplete = this._getAutocomplete(field);
        autocomplete.clearByButton();
        return this;
    }

    _getAutocomplete(field) {
        return new XlWidgetAutocomplete(`#${field}`);
    }

    _getSelect(model) {
        return element(by.$(`${this.path}`))
            .element(by.model(model));
    }

    _getSelectedText(model) {
        return this._getSelect(model)
            .element(by.css('option:checked'))
            .getText();
    }

    _selectOptionByTitle(model, title) {
        const option = this._getSelect(model)
            .element(by.cssContainingText('option', title));

        browser.wait(protractor.ExpectedConditions.elementToBeClickable(option), 10000);
        return option.click();
    }
}

class XlrDateWithVariables extends XlrInputWithVariables {
    constructor(path) {
        super(path, function () {
            return new DateTimePicker(`${path} .date-editor:first`);
        });
        this.path = path;
    }
}


global.XlrInputWithVariables = XlrInputWithVariables;
global.XlrInlinePasswordWithVariables = XlrInlinePasswordWithVariables;
global.XlrListStringWithVariables = XlrListStringWithVariables;
global.XlrSetStringWithVariables = XlrSetStringWithVariables;
global.XlrMapStringStringWithVariables = XlrMapStringStringWithVariables;
global.XlrIntegerWithVariables = XlrIntegerWithVariables;
global.XlrBooleanWithVariables = XlrBooleanWithVariables;
global.XlrDependencyWithVariables = XlrDependencyWithVariables;
global.XlrDateWithVariables = XlrDateWithVariables;
