let JIRA_ISSUE_API_URL = 'https://digitalai.atlassian.net/rest/api/2/issue';

let configurationItemsIds = undefined;
let usernames = undefined;
let releaseIds = undefined;
let archivedReleaseIds = undefined;
let globalVariableIds = undefined;
let folderIds = undefined;
let sharedConfigurationIds = [];
let userProfiles = undefined;
let riskProfiles = [];
let triggerIds = [];

class Fixtures {
    static initClass() {
        configurationItemsIds = [];
        sharedConfigurationIds = [];
        usernames = [];
        releaseIds = [];
        archivedReleaseIds = [];
        globalVariableIds = [];
        folderIds = [];
        userProfiles = [];
        riskProfiles = [];
        triggerIds = [];
    }

    static getParentId(id) {
        if (id.indexOf('/') === -1) return null;
        return id.substring(0, id.lastIndexOf('/'));
    }

    static sendRequest(type, url, data) {
        return Flow.execute((fulfill, reject) => Server.doRequest(type, url, data).then(fulfill, (error) => reject(error)));
    }

    static sendRequestAndIgnoreResult(type, url, data) {
        return Flow.execute((fulfill) => Server.doRequest(type, url, data).then(fulfill, fulfill));
    }

    static deleteCI(id) {
        return this.sendRequest('DELETE', `fixtures/configuration/${id}`);
    }

    static release(release) {
        initReleaseDefaults(release);
        releaseIds.push(release.id);
        return this.sendRequest('POST', 'fixtures/release', release);
    }

    static trigger(trigger) {
        triggerIds.push(trigger.id);
        return this.sendRequest('POST', 'fixtures/trigger', trigger);
    }

    static deleteTrigger(triggerId) {
        return this.sendRequest('DELETE', `fixtures/trigger/${triggerId}`);
    }

    static clearTriggers() {
        this.getAllTriggers().each((trigger) => {
            return this.deleteTrigger(trigger.triggerId);
        });
        triggerIds = [];
    }

    static archivedRelease(release) {
        initReleaseDefaults(release);
        archivedReleaseIds.push(release.id);
        return this.sendRequest('POST', 'fixtures/archived', release);
    }

    static preArchiveRelease(releaseId) {
        return this.sendRequest('GET', `fixtures/preArchive/${releaseId}`);
    }

    static archiveRelease(releaseId) {
        return this.sendRequest('GET', `fixtures/archive/${releaseId}`);
    }

    static markReleaseToDelete(id) {
        releaseIds.push(id);
    }

    static ci(ci) {
        configurationItemsIds.push(ci.id);
        return this.sendRequest('POST', 'fixtures', [ci]);
    }

    static configuration(ci) {
        sharedConfigurationIds.push(ci.id);
        return this.sendRequest('POST', 'fixtures/shared', [ci]);
    }

    static cis(cis) {
        for (let ci of cis) {
            configurationItemsIds.push(ci.id);
        }
        return this.sendRequest('POST', 'fixtures', cis);
    }

    static activityLogs(releaseId, logs) {
        return this.sendRequest('POST', `fixtures/activityLogs/${releaseId}`, logs);
    }

    static deleteRelease(releaseId) {
        if (releaseId.indexOf('Applications/') === -1) {
            releaseId = 'Applications/' + releaseId;
        }
        return this.sendRequest('DELETE', `fixtures/${releaseId}`);
    }

    static deleteArchivedRelease(id) {
        if (id.indexOf('Applications/') === -1) {
            id = `Applications/${id}`;
        }
        return this.sendRequest('DELETE', 'fixtures/cis', [
            {
                id,
                type: 'xlrelease.Release',
            },
        ]);
    }

    static addSystemTeams(teams) {
        const systemTeams = ['Folder Owner', 'Release Admin', 'Template Owner'];
        const teamsToUpdate = teams.map((team) => team.teamName);
        systemTeams.forEach((teamName) => {
            if (teamsToUpdate.indexOf(teamName) === -1) {
                teams.push({
                    type: 'xlrelease.Team',
                    teamName,
                    members: [],
                    permissions: [],
                });
            }
        });
        return teams;
    }

    static folder(folderJson) {
        let parentId = Fixtures.getParentId(folderJson.id);
        folderJson.type = 'xlrelease.Folder';

        let teams = folderJson.teams;
        delete folderJson.teams;

        let children = folderJson.children || [];
        delete folderJson.children;

        return this.sendRequest('POST', `api/v1/folders/${parentId}`, folderJson)
            .then(() => {
                folderIds.push(folderJson.id);
                if (teams) {
                    teams = this.addSystemTeams(teams);
                    return this.sendRequest('POST', `api/v1/folders/${folderJson.id}/teams`, teams);
                } else {
                    return Promise.resolve(null);
                }
            })
            .then(() => {
                if (children && children.length > 0) {
                    return Promise.all(_.map(children, (child) => this.folder(child)));
                } else {
                    return Promise.resolve(undefined);
                }
            });
    }

    static roles(rolePrincipals) {
        return this.sendRequest('POST', 'fixtures/roles/principals', rolePrincipals);
    }

    static deleteRole(roleName) {
        return this.sendRequest('DELETE', `fixtures/role/${roleName}`);
    }

    static permissions(rolePermissions) {
        return this.sendRequest('POST', 'fixtures/roles/permissions/global', rolePermissions);
    }

    static addUser(username, password) {
        usernames.push(username);
        return this.sendRequest('POST', 'fixtures/user', { username, password });
    }

    static deleteUser(username) {
        return this.sendRequest('DELETE', `fixtures/user/${username}`);
    }

    static taskAccesses(taskAccesses) {
        return this.sendRequest('PUT', 'tasks/types-access', taskAccesses);
    }

    static deleteTaskAccesses() {
        return this.sendRequest('DELETE', 'fixtures/task-access');
    }

    static addDefaultUser() {
        return this.addUser('Itchy', 'Itchy');
    }

    static addUserProfile(username, profile) {
        if (!profile) {
            profile = {};
        }
        profile.id = username;
        profile.type = 'xlrelease.UserProfile';
        profile.canonicalId = username.toLowerCase();
        userProfiles.push(profile.canonicalId);
        return this.sendRequest('POST', 'fixtures/userProfile', profile);
    }

    static deleteUserProfile(username) {
        return this.sendRequest('DELETE', `fixtures/userProfile/${username}`);
    }

    static updateUserProfile(username, profile) {
        profile.id = username;
        profile.type = 'xlrelease.UserProfile';
        profile.canonicalId = username.toLowerCase();
        if (userProfiles.indexOf(profile.canonicalId) === -1) {
            userProfiles.push(profile.canonicalId);
        }

        return this.sendRequest('PUT', 'fixtures/userProfile', profile);
    }

    static resetDefaultUserProfile() {
        this.updateUserProfile('Itchy', { email: 'itchy@default.com', loginAllowed: true });
    }

    static expectUserProfileExists(username, userEmail, fullName) {
        return this.sendRequest('GET', `fixtures/userProfile?name=${username}&email=${userEmail}&fullName=${fullName}`);
    }

    static expectContainingAttachments(releaseId, tempFile) {
        let ok = function () {};
        let failure = function (message) {
            throw `Can't find attachment '${tempFile.getName()}' with content : '${tempFile.getContent()}'. Message from server: '${message}'`;
        };
        if (releaseId.indexOf('Applications/') === -1) {
            releaseId = 'Applications/' + releaseId;
        }
        return this.sendRequest('POST', `fixtures/expectContainingAttachments/${releaseId}`, {
            name: tempFile.getName(),
            expectedContent: tempFile.getContent(),
        }).then(ok, failure);
    }

    static addJiraTask() {
        return Flow.execute(function (fulfill, reject) {
            let data = {
                fields: {
                    project: { key: 'RIT' },
                    summary: 'Test Task',
                    description: 'Test task description',
                    issuetype: { name: 'Task' },
                },
            };
            let options = {
                method: 'POST',
                body: data,
                uri: JIRA_ISSUE_API_URL,
                json: true,
                auth: { username: 'xlr-jira-testuser@xebialabs.com', password: 'zgsXK6c3oLVkJxrWVJQy9DB7', sendImmediately: true },
                resolveWithFullResponse: true,
            };
            return requestPromise(options)
                .then((result) => fulfill(result.body['key']))
                .catch((result) => reject(result.error));
        });
    }

    static expectJiraTaskStatus(taskId, expectedStatus) {
        return Flow.execute(function (fulfill, reject) {
            let options = {
                method: 'GET',
                uri: `${JIRA_ISSUE_API_URL}/${taskId}`,
                json: true,
                auth: { username: 'xlr-jira-testuser@xebialabs.com', password: 'zgsXK6c3oLVkJxrWVJQy9DB7', sendImmediately: true },
                resolveWithFullResponse: true,
            };
            return requestPromise(options)
                .then(function (result) {
                    let { body } = result;
                    let status = body.fields.status.name;
                    if (status !== expectedStatus) {
                        reject(`Expected task ${taskId} to be '${expectedStatus}' but was '${status}'. Message from server : ${body}`);
                    }
                    return fulfill();
                })
                .catch((result) => reject(result.error));
        });
    }

    static expectJiraTaskSummary(taskId, expectedSummary) {
        return Flow.execute(function (fulfill, reject) {
            let options = {
                method: 'GET',
                uri: `${JIRA_ISSUE_API_URL}/${taskId}`,
                json: true,
                auth: { username: 'xlr-jira-testuser@xebialabs.com', password: 'zgsXK6c3oLVkJxrWVJQy9DB7', sendImmediately: true },
                resolveWithFullResponse: true,
            };
            return requestPromise(options)
                .then(function (result) {
                    let { body } = result;
                    let { summary } = body.fields;
                    if (summary !== expectedSummary) {
                        reject(`Expected task ${taskId} summary to be '${expectedSummary}' but was '${summary}'. Message from server : ${body}`);
                    }
                    return fulfill();
                })
                .catch((result) => reject(result.error));
        });
    }

    static globalVariable(variable) {
        variable.id = variable.id || `Configuration/variables/global/Variable_${variable.key}`;
        globalVariableIds.push(variable.id);
        variable.requiresValue = false;
        variable.showOnReleaseStart = false;
        return this.sendRequest('POST', 'api/v1/config/Configuration/variables/global', variable);
    }

    static deleteGlobalVariable(variableId) {
        return this.sendRequestAndIgnoreResult('DELETE', `api/v1/config/${variableId}`);
    }

    static deleteAllGlobalVariables() {
        return this.sendRequest('DELETE', 'fixtures/globalVariables');
    }

    static clearCalendar() {
        return this.sendRequest('DELETE', 'fixtures/calendar');
    }

    static addBlackout(name, from, to) {
        return this.sendRequest('POST', 'calendar/blackouts', {
            label: name,
            startDate: from.toDate(),
            endDate: to.toDate(),
            type: 'xlrelease.Blackout',
            id: null,
        });
    }

    static deleteFolder(folderId) {
        return this.sendRequest('GET', `api/v1/folders/${folderId}`).then(
            () => this.sendRequest('DELETE', `api/v1/folders/${folderId}`),
            () => {
                return Promise.resolve(undefined);
            },
        );
    }

    static deleteSharedConfigurations(configurationIds) {
        return this.sendRequest('DELETE', 'fixtures/shared', [configurationIds]);
    }

    static riskProfile(riskProfile) {
        return this.sendRequest('POST', 'fixtures/riskProfiles', riskProfile).then(() => riskProfiles.push(riskProfile.id));
    }

    static deleteRiskProfile(riskProfileId) {
        return this.sendRequest('DELETE', `fixtures/riskProfiles/${riskProfileId}`);
    }

    static clean() {
        for (let username of usernames) {
            this.deleteUser(username);
        }
        for (let triggerId of triggerIds) {
            this.deleteTrigger(triggerId);
        }
        for (let releaseId of releaseIds.reverse()) {
            this.deleteRelease(releaseId);
        }
        for (let releaseId of archivedReleaseIds) {
            this.deleteArchivedRelease(releaseId);
        }
        for (let folderId of folderIds.reverse()) {
            this.deleteFolder(folderId);
        }
        for (let configurationItemId of configurationItemsIds) {
            this.deleteCI(configurationItemId);
        }
        for (let variableId of globalVariableIds) {
            this.deleteGlobalVariable(variableId);
        }
        for (let sharedId of sharedConfigurationIds) {
            this.deleteSharedConfigurations(sharedId);
        }
        for (let userProfile of userProfiles) {
            this.deleteUserProfile(userProfile);
        }
        for (let riskProfile of riskProfiles) {
            this.deleteRiskProfile(riskProfile);
        }
        releaseIds = [];
        folderIds = [];
        configurationItemsIds = [];
        triggerIds = [];
        usernames = [];
        globalVariableIds = [];
        sharedConfigurationIds = [];
        userProfiles = [];
        riskProfiles = [];
    }
}
Fixtures.initClass();

global.fixtures = () => Fixtures;
