class InlineEditor {
    constructor(path) {
        this.path = path;
        Browser.waitFor(this.path);
        this.inlineEditor = element.all(By.$(this.path)).first();
    }

    set(text, withScroll = true) {
        if (withScroll) {
            browser.executeScript("arguments[0].scrollIntoView();", this.inlineEditor.getWebElement());
        }
        this.focus(withScroll);
        this.enter(text);
        return this.blur();
    }

    focus(withScroll = true) {
        if (withScroll) {
            browser.executeScript("arguments[0].scrollIntoView();", this.inlineEditor.getWebElement());
        }
        this.inlineEditor.element(By.css(".display")).click();
        return this;
    }

    deselect() {
        this.inlineEditor.element(By.css(".editable")).sendKeys(protractor.Key.LEFT);
        return this;
    }

    enter(text) {
        Browser.waitFor(`${this.path} .editable`, 3000);
        return this.inlineEditor.element(By.css(".editable")).sendKeys(text);
    }

    select(value) {
        this.focus();
        this.inlineEditor.element(By.$(`option:contains('${value}')`)).click();
        return this.blur();
    }

    expectOptionNotPresent(value) {
        this.focus();
        expect(element(By.$(`${this.path} option:contains('${value}')`)).isPresent()).toBe(false);
        return this.blur();
    }

    value() {
        return this.inlineEditor.element(By.css("span.text span")).getText();
    }

    editorValue() {
        return this.inlineEditor.element(By.css(".editable")).getAttribute('value');
    }

    blur() {
        Browser.clickElseWhere();
        return this;
    }
}


class AutoCompleteInlineEditor extends InlineEditor {
    set(value) {
        this.focus();
        this.enter(value);
        return this.chooseCandidate(value);
    }

    chooseCandidate(candidate) {
        Browser.waitFor(`.ui-autocomplete:visible .ui-menu-item:contains('${candidate}')`);
        return element(By.$(`.ui-autocomplete:visible .ui-menu-item:contains('${candidate}')`)).click();
    }
}

class UserInlineEditor extends AutoCompleteInlineEditor {
    enter(text) {
        return this.inlineEditor.element(by.model('draft.username')).sendKeys(text);
    }
}

class TextareaInlineEditor extends InlineEditor {
    focus() {
        // relies on a click on a span::before (not actionnable directly by Protractor)
        let clickOnEditIcon = function () {
            return $(`${arguments[0]} .edit-icon`).click();
        };
        Browser.executeVoidScript(clickOnEditIcon, this.path);
        return this;
    }

    set(text) {
        this.focus();
        this.enter(text);
        return this.inlineEditor.element(By.css('.ok')).click();
    }
}

class CodeInlineEditor extends InlineEditor {
    focus() {
        // relies on a click on a span::before (not actionnable directly by Protractor)
        let clickOnEditIcon = function () {
            return $(`${arguments[0]} .edit-icon`).click();
        };
        Browser.executeVoidScript(clickOnEditIcon, this.path);
        browser.actions().mouseMove(element(By.css('div.ace_content'))).perform();
        browser.actions().click().perform();
        return this;
    }

    fullScreen() {
        this.focus();
        element(by.css('.ace-script-editor-full-screen-mode-icon')).click();
        return element(by.css('.ace-script-editor.full-screen'));
    }

    clear() {
        this.focus();
        element(By.css(`.ace_content`)).getText().then((text)=>{
            for(let i = 0; i < text.length; i++) {
                element(By.css(`.ace_text-input`)).sendKeys(protractor.Key.BACK_SPACE);
            }
        });
        return this.blur();
    }

    set(text) {
        this.clear();
        this.enter(text);
        browser.actions().sendKeys(protractor.Key.ESCAPE).perform();
        return this.inlineEditor.element(By.css('.ok')).click();
    }

    enter(text) {
        return element(By.css('textarea.ace_text-input')).clear().sendKeys(text);
    }

    value() {
        return browser.executeScript(() => {
            return $('.ace-script-editor-code').data('$ngModelController').$viewValue;
        });
    }
}

global.InlineEditor = InlineEditor;
global.TextareaInlineEditor = TextareaInlineEditor;
global.AutoCompleteInlineEditor = AutoCompleteInlineEditor;
global.UserInlineEditor = UserInlineEditor;
global.CodeInlineEditor = CodeInlineEditor;
