package com.xebialabs.xlrelease.ascode.service.spec

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.service.spec.{InterpreterContext, SpecInterpreter}
import com.xebialabs.ascode.utils.TypeSugar.typeOf
import com.xebialabs.ascode.utils.{Utils => PlatformAsCodeUtils}
import com.xebialabs.ascode.yaml.dto.AsCodeResponse
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.ChangedIds
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.permission.PlatformPermissions
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.service._
import com.xebialabs.xlrelease.ascode.service.spec.CiSpecInterpreter.{CisByScopeWithActions, ScopeAndActions}
import com.xebialabs.xlrelease.ascode.utils.{FolderScope, GlobalScope, ImportContext, ImportScope, Utils => XlrAsCodeUtils}
import com.xebialabs.xlrelease.ascode.yaml.model.{CalendarAsCode, EmailNotificationSettingsAsCode}
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.variables.{FolderVariables, GlobalVariables}
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.{BaseConfiguration, Configuration, Release, Trigger}
import com.xebialabs.xlrelease.notifications.actor.NotificationServiceActor.ClearCache
import com.xebialabs.xlrelease.notifications.actor.NotificationServiceActorHolder
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.security.{PermissionChecker, XLReleasePermissions}
import com.xebialabs.xlrelease.versioning.ascode.{ImportValidator}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.immutable.ListMap
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.reflect.ClassTag

object CiSpecInterpreter {
  case class ProcessedCi[T <: ConfigurationItem](ci: T, changedIds: ChangedIds, postCommitActions: Seq[PostCommitAction] = Seq.empty)

  case class CisByScopeWithActions(cisByScope: ListMap[ImportScope, List[ConfigurationItem]] = ListMap.empty,
                                   postCommitActions: Seq[PostCommitAction] = Seq.empty)

  case class ScopeAndActions(scope: ImportScope, postCommitActions: Seq[PostCommitAction] = Seq.empty)
}

@Component
@Autowired
class CiSpecInterpreter(folderService: FolderAsCodeService,
                        metadataProcessor: MetadataProcessor,
                        templateService: TemplateAsCodeService,
                        patternService: DeliveryPatternAsCodeService,
                        configurationService: ConfigurationAsCodeService,
                        settingsAsCodeService: SettingsAsCodeService,
                        dashboardAsCodeService: DashboardAsCodeService,
                        riskProfileAsCodeService: RiskProfileAsCodeService,
                        triggerAsCodeService: TriggerAsCodeService,
                        notificationsAsCodeService: NotificationSettingsAsCodeService,
                        calendarAsCodeService: CalendarAsCodeService,
                        folderVersioningAsCodeService: FolderVersioningAsCodeService,
                        globalVariableAsCodeService: GlobalVariableAsCodeService,
                        folderVariableAsCodeService: FolderVariableAsCodeService,
                        permissionsChecker: PermissionChecker,
                        notificationServiceActorHolder: NotificationServiceActorHolder)
  extends SpecInterpreter with Logging with DefinitionImporter {

  private lazy val notificationServiceActor = notificationServiceActorHolder.actorRef()

  override def isDefinedAt(context: InterpreterContext): Boolean = context.definition.spec.isInstanceOf[CiSpec]

  override def apply(context: InterpreterContext): AsCodeResponse = {
    val result = importDefinition(ImportIncludes(), context.definition, context.scmTraceabilityData, None)

    // Not in a transaction here so just run all the actions now
    result.postCommitActions.foreach(action => action.run())

    AsCodeResponse.ids(result.changedIds: _*)
  }

  override def importDefinition(importIncludes: ImportIncludes, definition: Definition, scmData: Option[SCMTraceabilityData], validator: Option[ImportValidator]): ImportResult = {
    val metadata = metadataProcessor.processMetadata(definition.kind, definition.metadata)
    val ciSpec = definition.spec.asInstanceOf[CiSpec]

    val cisByScopeAndActions = createFoldersAndMapCis(metadata, ciSpec)
    val flattened = cisByScopeAndActions.cisByScope.toList.filter { case (_, cis) => cis.nonEmpty }
    // pre-process templates for create release tasks
    val templates = flattened
      .foldLeft(ListMap.empty[ImportScope, List[Release]]) {
        case (map, (scope, cis)) =>
          val onlyTemplates = cis.filter { ci => ci.isInstanceOf[Release] }.map(_.asInstanceOf[Release])

          if (onlyTemplates.nonEmpty) {
            map + (scope -> onlyTemplates)
          } else {
            map
          }
      }

    val templateIdsByPaths = templateService.findOrPredictTemplateIds(templates)
    val importResult = flattened.foldLeft(ImportResult(List(CI.ids), cisByScopeAndActions.postCommitActions)) { case (acc, (scope, cis)) =>
      val localCtx = ImportContext(scope, metadata, ciSpec.references, scmData, templateIdsByPaths, validator)
      val (changedIdsPerType, ir) = importCisInDbForContext(importIncludes, localCtx, cis)
      acc.merge(
        scope.getFolderId.fold(ir)(fid =>
          ir.copy(changedIdsPerTypePerFolder = Map(fid -> changedIdsPerType))
        )
      )
    }

    importResult
  }

  private def typeHandler[T <: ConfigurationItem : ClassTag](handle: T => ChangedIds): PartialFunction[ConfigurationItem, ChangedIds] = {
    case ci if ci.getType.instanceOf(typeOf[T]) =>
      logger.debug(s"Processing the following CI: $ci)")
      handle(ci.asInstanceOf[T])
  }

  private def importCisInDbForContext(includes: ImportIncludes, ctx: ImportContext, cis: Seq[ConfigurationItem]): (Map[Type, ChangedIds], ImportResult) = {
    ctx.scope match {
      case f: FolderScope => permissionsChecker.checkAny(f.id, XLReleasePermissions.APPLY_FOLDER_CHANGES, PlatformPermissions.ADMIN)
      case GlobalScope => permissionsChecker.check(PlatformPermissions.ADMIN)
      case _ =>
    }

    val sortedCis = cis.sortWith(XlrAsCodeUtils.ciSortOrderValue(_) < XlrAsCodeUtils.ciSortOrderValue(_))

    var clearNotifications = false

    val changedIdsPerType = mutable.Map.empty[Type, ChangedIds]
    val result = sortedCis.foldLeft(ImportResult(List(CI.ids))) { case (acc, unprocessedCi) =>
      unprocessedCi match {
        case _: EmailNotificationSettingsAsCode => clearNotifications = true
        case _ =>
      }
      val (tpe, importResult) = importCiInDbForContext(includes, ctx, unprocessedCi)
      val changedIds = importResult.changedIds.foldLeft(CI.ids)(_.merge(_))

      changedIdsPerType.get(tpe) match {
        case Some(existing) => changedIdsPerType.update(tpe, existing.merge(changedIds))
        case None => changedIdsPerType.update(tpe, changedIds)
      }

      acc.merge(importResult)
    }

    changedIdsPerType.toMap -> (if (clearNotifications) {
      result.withPostCommitAction(() => notificationServiceActor ! ClearCache())
    } else {
      result
    })
  }

  //noinspection ScalaStyle
  private def importCiInDbForContext[T <: ConfigurationItem](includes: ImportIncludes, importContext: ImportContext, ci: T): (Type, ImportResult) = {
    def processConditionally[CI <: ConfigurationItem : ClassTag](condition: Boolean, processFn: => ImportResult) = {
      typeOf[CI] -> (if (condition) {
        processFn
      } else {
        ImportResult.empty
      })
    }

    ci match {
      case template: Release =>
        processConditionally[Release](includes.templates, templateService.process(importContext, template))
      case configuration: Configuration =>
        processConditionally[Configuration](includes.configurations, configurationService.process(importContext, configuration))
      case globalVariables: GlobalVariables =>
        processConditionally[GlobalVariables](includes.variables, globalVariables.getVariables.asScala.foldLeft(ImportResult(List(CI.ids))) {
          case (ir, variable) => ir.merge(globalVariableAsCodeService.process(importContext, variable))
        })
      case folderVariables: FolderVariables =>
        processConditionally[FolderVariables](includes.variables, folderVariables.getVariables.asScala.foldLeft(ImportResult(List(CI.ids))) {
          case (ir, variable) => ir.merge(folderVariableAsCodeService.process(importContext, variable))
        })
      case trigger: Trigger =>
        processConditionally[Trigger](includes.triggers, triggerAsCodeService.process(importContext, trigger))
      case pattern: Delivery =>
        processConditionally[Delivery](includes.patterns, patternService.process(importContext, pattern))
      case notifSettings: EmailNotificationSettingsAsCode =>
        processConditionally[EmailNotificationSettingsAsCode](includes.notifications, notificationsAsCodeService.process(
          importContext, EmailNotificationSettingsAsCode.toEmailNotificationSettings(notifSettings)
        ))
      case dashboard: Dashboard =>
        processConditionally[Dashboard](includes.dashboards, dashboardAsCodeService.process(importContext, dashboard))
      case _ =>
        ci.getType -> importCiInDbForContextLegacy(importContext, ci)
    }
  }

  private def importCiInDbForContextLegacy[T <: ConfigurationItem](importContext: ImportContext, ci: T): ImportResult = {
    val (folderId, folderPath) = importContext.scope match {
      case f: FolderScope => (Some(f.id), Some(f.path))
      case _ => (None, None)
    }

    val processor = typeHandler[CalendarAsCode](calendarAsCodeService.updateCalendarEntries)
      .orElse(typeHandler[RiskProfile](riskProfileAsCodeService.createOrUpdateRiskProfile))
      .orElse(typeHandler[FolderVersioningSettings](folderVersioningAsCodeService.createConfiguration(_, importContext.references, folderId, folderPath)))
      .orElse(typeHandler[BaseConfiguration](settingsAsCodeService.updateSettings))
      .orElse[ConfigurationItem, ChangedIds] { case ci =>
        throw new AsCodeException(s"There is no service available for this type of ci [${ci.getType.toString}].")
      }

    ImportResult(List(processor(ci)))
  }

  private def mergeCisByScopeWithActions(obj1: CisByScopeWithActions, obj2: CisByScopeWithActions): CisByScopeWithActions = {
    val map1 = obj1.cisByScope
    val map2 = obj2.cisByScope
    val cisByScope = map1 ++ map2.map {
      case (key, value) => key -> map1.get(key).map(_ ++ value).getOrElse(value)
    }
    val actions = obj1.postCommitActions ++ obj2.postCommitActions
    CisByScopeWithActions(cisByScope, actions)
  }

  private def getTopLevelScope(metadata: Map[String, String]): ImportScope = {
    val home = metadata.get(MetadataFields.HOMEFOLDER.toString)
    val folderId = metadata.get(MetadataFields.FOLDER.toString)

    folderId match {
      case None => GlobalScope
      case Some(id) =>
        val folderRow = folderService.getFolderRow(id)
        FolderScope(id, folderService.getFolderPath(id), folderRow.uid, folderRow.securityUid, home)
    }
  }

  private def buildFolderPath(parentScope: ImportScope, folderTitle: String): String = {
    parentScope.getFolderPath match {
      case None => folderTitle
      case Some(path) => XlrAsCodeUtils.joinPaths(Seq(path, folderTitle))
    }
  }

  private def findOrCreateFolder(parentScope: ImportScope, folderTitle: String): ScopeAndActions = {
    val parentId = parentScope.getFolderId.getOrElse(ROOT_FOLDER_ID)
    val absolutePath = buildFolderPath(parentScope, folderTitle)
    val rowAndActions = folderService.findSubFolderRowByTitle(folderTitle, parentId) match {
      case None =>
        val result = folderService.createFolderInTransaction(folderTitle, parentId)
        (folderService.getFolderRow(result.folder.getId), result.postCommitActions)
      case Some(existing) => (existing, Seq.empty[PostCommitAction])
    }
    val row = rowAndActions._1
    val actions = rowAndActions._2
    ScopeAndActions(
      FolderScope(row.folderId.absolute, absolutePath, row.uid, row.securityUid, parentScope.getMetadataHome),
      actions
    )
  }

  private def findOrCreateFolders(parentScope: ImportScope, path: String): List[ScopeAndActions] = {
    // YAML has supported defining a folder hierarchy so there may be multiple folders to find/create
    val folders = PlatformAsCodeUtils.splitStringByPathSeparator(path)

    val scopesAndActions = folders.foldLeft(List(ScopeAndActions(parentScope))) {
      case (acc, title) => {
        acc :+ findOrCreateFolder(acc.last.scope, title)
      }
    }

    scopesAndActions.tail // strip off the parent scope
  }

  private def initializeCisByScope(scopes: List[ImportScope]): ListMap[ImportScope, List[ConfigurationItem]] = {
    scopes.foldLeft(ListMap.empty[ImportScope, List[ConfigurationItem]]) {
      (acc, scope) => {
        acc + (scope -> List.empty[ConfigurationItem])
      }
    }
  }

  private def createFoldersAndMapCis(metadata: Map[String, String], spec: CiSpec): CisByScopeWithActions = {
    val scope = getTopLevelScope(metadata)
    createFoldersAndMapCis(scope, spec.cis)
  }

  private def createFoldersAndMapCis(parentScope: ImportScope, childCis: List[ConfigurationItem]): CisByScopeWithActions = {
    val (subfolders, cis) = childCis.partition(_.isInstanceOf[Folder])

    subfolders.map(_.asInstanceOf[Folder]).foldLeft(CisByScopeWithActions(ListMap(parentScope -> cis))) {
      (map, subfolder) => {
        val scopesAndActions = findOrCreateFolders(parentScope, subfolder.getTitle)
        val folderActions = scopesAndActions.map(_.postCommitActions).foldLeft(Seq.empty[PostCommitAction]) {
          case (acc, actions) => acc ++ actions
        }
        val emptyParentsAndAllActions = CisByScopeWithActions(
          initializeCisByScope(scopesAndActions.map(_.scope).slice(0, scopesAndActions.length - 1)),
          folderActions
        )
        val children = subfolder.getChildren.asInstanceOf[java.util.Set[ConfigurationItem]].asScala.toList
        mergeCisByScopeWithActions(mergeCisByScopeWithActions(map, emptyParentsAndAllActions), createFoldersAndMapCis(scopesAndActions.last.scope, children))
      }
    }
  }
}
