package com.xebialabs.xlrelease.ascode.service.generatestrategy

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.utils.Utils.filterFolderOnGeneratePermissions
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.variables.{FolderVariables, GlobalVariables, Variable}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.service.{FolderVariableService, VariableService}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util.stream.Collectors
import scala.collection.immutable.ListMap
import scala.collection.mutable.ListBuffer

@Component
class VariableAsCodeGenerator @Autowired()(variableService: VariableService,
                                           folderVariableService: FolderVariableService,
                                           permissions: PermissionChecker) extends GenericGenerateStrategy {
  /**
   * This order will be used to sort generate strategies
   */
  override val generateOrder: Double = 5

  override def isDefinedAt(config: CiGenerateConfig): Boolean = {
    config.generateConfig.generateVariables
  }

  override protected def generateFolder(config: CiGenerateConfig): ListMap[Option[Folder], List[ConfigurationItem]] = {
    val variablesByFolders = config.folders
      .filter { folder =>
        filterFolderOnGeneratePermissions(config, folder.getId) ||
          permissions.hasPermission(VIEW_TEMPLATE, folder.getId) ||
          permissions.hasPermission(VIEW_RELEASE, folder.getId) ||
          permissions.hasPermission(VIEW_DASHBOARD, folder.getId) ||
          permissions.hasPermission(EDIT_FOLDER_VARIABLES, folder.getId)
      }
      .map { folder =>
        val configs = ListBuffer.empty[ConfigurationItem]
        val folderVars = folderVariableService.getAllFromImmediateParent(folder.getId)
        sugarizeFolderAndGlobalVarProps(folderVars.getVariables)
        if (!folderVars.getVariables.isEmpty) {
          configs += folderVars
        }

        Some(folder) -> configs.toList
      }

    ListMap(variablesByFolders: _*)
  }

  override protected def generateGlobal(config: CiGenerateConfig): ListMap[Option[Folder], List[ConfigurationItem]] = {
    val configs = ListBuffer.empty[ConfigurationItem]

    val globalVars = new GlobalVariables()
    globalVars.setVariables(variableService.findGlobalVariablesOrEmpty().getVariables)
    sugarizeFolderAndGlobalVarProps(globalVars.getVariables)
    if (!globalVars.getVariables.isEmpty) {
      configs += globalVars
    }

    ListMap(None -> configs.toList)
  }

  override protected def filter(results: ListMap[Option[Folder], List[ConfigurationItem]], ciTitle: String):
  ListMap[Option[Folder], List[ConfigurationItem]] = {
    results.map { case (maybeFolder, cis) =>
      val filtered = cis.headOption.fold(List.empty[ConfigurationItem]) { ci =>
        maybeFolder match {
          case Some(_) =>
            val fvar = ci.asInstanceOf[FolderVariables]
            fvar.setVariables(filterVariables(fvar.getVariables, ciTitle))
            if (fvar.getVariables.isEmpty) List.empty else List(fvar)
          case None =>
            val gvar = ci.asInstanceOf[GlobalVariables]
            gvar.setVariables(filterVariables(gvar.getVariables, ciTitle))
            if (gvar.getVariables.isEmpty) List.empty else List(gvar)
        }
      }

      maybeFolder -> filtered
    }
  }

  private def sugarizeFolderAndGlobalVarProps(vars: java.util.List[Variable]): Unit = {
    // hack: set to true so that the properties aren't visible in YAML for global and folder variables
    vars.forEach { variable =>
      variable.setShowOnReleaseStart(true)
      variable.setRequiresValue(true)
    }
  }

  private def filterVariables(vars: java.util.List[Variable], ciTitle: String) = {
    vars.stream
      .filter(v => Utils.stringLike(v.getKey, ciTitle))
      .collect(Collectors.toList[Variable])
  }
}
