package com.xebialabs.xlrelease.domain.versioning.ascode

import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION
import com.xebialabs.deployit.plugin.api.udm.{Metadata, Property}
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings.CATEGORY_INTERNAL
import com.xebialabs.xlrelease.scm.connector.{Repository, ScmException}
import com.xebialabs.xlrelease.scm.data.ScmUtils.normalize
import com.xebialabs.xlrelease.variable.VariableHelper
import org.eclipse.jgit.lib.{Repository => GitRepositoryPlugin}

import java.beans.BeanProperty
import java.util
import java.util.Date
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

object FolderVersioningSettings {
  val FOLDER_VERSIONING_ID_ROOT: String = CONFIGURATION.getRootNodeName + "/versioning"
  val FOLDER_VERSIONING_YAML_FILENAME = "Releasefile.yaml"
  val CATEGORY_INTERNAL = "internal"

  val TEMPLATES_VERSIONING_YAML_FILENAME = "Templates.yaml"
  val PATTERNS_VERSIONING_YAML_FILENAME = "Patterns.yaml"
  val TRIGGERS_VERSIONING_YAML_FILENAME = "Triggers.yaml"
  val VARIABLES_VERSIONING_YAML_FILENAME = "Variables.yaml"
  val PERMISSIONS_VERSIONING_YAML_FILENAME = "Permissions.yaml"
  val NOTIFICATIONS_VERSIONING_YAML_FILENAME = "Notifications.yaml"
  val DASHBOARDS_VERSIONING_YAML_FILENAME = "Dashboards.yaml"
  val CONNECTIONS_VERSIONING_YAML_FILENAME = "Connections.yaml"

  val ALL_FILENAMES = Seq(FOLDER_VERSIONING_YAML_FILENAME,
    TEMPLATES_VERSIONING_YAML_FILENAME,
    TRIGGERS_VERSIONING_YAML_FILENAME,
    VARIABLES_VERSIONING_YAML_FILENAME,
    PERMISSIONS_VERSIONING_YAML_FILENAME,
    NOTIFICATIONS_VERSIONING_YAML_FILENAME,
    DASHBOARDS_VERSIONING_YAML_FILENAME,
    CONNECTIONS_VERSIONING_YAML_FILENAME)

  def getDefinitionsPath(versioningConfig: FolderVersioningSettings, filename: String = FOLDER_VERSIONING_YAML_FILENAME): String = {
    s"${normalize(versioningConfig.scmPath)}/$filename".replaceAll("/+", "/")
  }

  def getfilename(filepath: String): String = {
    if (filepath.contains("/")) {
      filepath.substring(filepath.lastIndexOf("/") + 1)
    } else {
      filepath
    }
  }

  def generateTagName(prefix: String,
                      version: String,
                      format: String = "${prefix}/${version}"): Try[String] = {
    if (version.isBlank) {
      Failure(ScmException(s"Unable to generate tag with config: prefix=[$prefix] version=[$version] format=[$format]"))
    } else {
      val tag = VariableHelper.replaceAll(
        format,
        java.util.Map.of("${prefix}", prefix, "${version}", version)
      ).strip().stripPrefix("/").stripSuffix("/")

      if (tag.isEmpty) {
        Failure(ScmException(s"Unable to generate tag from configured scmPath"))
      } else {
        Success(tag)
      }
    }
  }
}

@Metadata(label = "As-code Versioning", description = "", virtual = false)
class FolderVersioningSettings extends BaseConfiguration {

  @BeanProperty
  @Property(
    label = "Git connection",
    description = "Git connection to use")
  var gitConnection: Repository = _

  @BeanProperty
  @Property(
    label = "SCM Path",
    description = "Path on the SCM where the as-code YAML is located")
  var scmPath: String = ""

  @BeanProperty
  @Property(
    required = true,
    label = "Branch",
    description = "Override branch setting")
  var branch = "master"

  @BeanProperty
  @Property(
    defaultValue = "true",
    description = "Export templates into the as-code YAML",
    category = "Export")
  var exportTemplates: Boolean = true

  @BeanProperty
  @Property(
    defaultValue = "true",
    description = "Export pattern into the as-code YAML",
    category = "Export")
  var exportPatterns: Boolean = true

  @BeanProperty
  @Property(
    defaultValue = "true",
    description = "Export variables into the as-code YAML",
    category = "Export")
  var exportVariables: Boolean = true

  @BeanProperty
  @Property(
    defaultValue = "true",
    description = "Export triggers into the as-code YAML",
    category = "Export")
  var exportTriggers: Boolean = true

  @BeanProperty
  @Property(
    defaultValue = "true",
    description = "Export connections into the as-code YAML",
    category = "Export")
  var exportConfiguration: Boolean = true

  @BeanProperty
  @Property(
    defaultValue = "true",
    description = "Export dashboards into the as-code YAML",
    category = "Export")
  var exportDashboards: Boolean = true

  @BeanProperty
  @Property(
    defaultValue = "true",
    description = "Export notifications into the as-code YAML",
    category = "Export")
  var exportNotifications: Boolean = true

  @BeanProperty
  @Property(
    defaultValue = "false",
    description = "Export teams and permissions into the as-code YAML",
    category = "Export")
  var exportSecurity: Boolean = false

  @BeanProperty
  @Property(category = "internal", required = false)
  var lastFetched: Date = _

  @BeanProperty
  @Property(category = "internal", required = false)
  var appliedVersion: String = ""

  @BeanProperty
  @Property(category = "internal", required = false)
  var appliedDate: Date = _

  @BeanProperty
  @Property(category = "internal", required = false)
  var appliedBy: String = ""

  @BeanProperty
  @Property(category = "internal", required = false, password = true)
  var secrets: java.util.Map[String, String] = new util.HashMap[String, String]()

  @BeanProperty
  @Property(defaultValue = "singleFile")
  var versioningStyle: String = ""

  def getEffectiveTagPrefix: String = {
    if (Strings.isBlank(scmPath)) {
      ""
    } else {
      GitRepositoryPlugin.normalizeBranchName(scmPath)
    }
  }

  def updateWith(newConfig: FolderVersioningSettings): Unit = {
    getEditableProperties().asScala.foreach { property =>
      property.set(this, property.get(newConfig))
    }
  }

  def getEditableProperties(): java.util.List[PropertyDescriptor] = {
    getType.getDescriptor.getPropertyDescriptors.asScala
      .filterNot(_.getCategory == CATEGORY_INTERNAL)
      .toList.asJava
  }
}
