package com.xebialabs.xlrelease.domain

import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service.{ExactSearch, GlobalSearch, SearchScope}
import com.xebialabs.xlrelease.repository.Ids

import java.util.{ArrayList => JArrayList, List => JList}
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._

object GlobalExportConfig {
  def apply(): GlobalExportConfig = new GlobalExportConfig()

  private def toSearchScope(exportConfig: GlobalExportConfig): SearchScope = {
    if (exportConfig.includeEntities.isEmpty) {
      GlobalSearch
    } else {
      ExactSearch(ids = exportConfig.includeEntities.asScala.toList)
    }
  }

  private def toCisConfig(exportConfig: GlobalExportConfig): CisConfig = {
    val outOfScope = false
    CisConfig(
      generatePermissions = exportConfig.includeKinds.permissions,
      generateRoles = exportConfig.includeKinds.roles,
      generateUsers = exportConfig.includeKinds.users,
      generateEnvironments = exportConfig.includeKinds.environments,
      generateApplications = exportConfig.includeKinds.applications,
      includeSecrets = false,
      generateTemplates = exportConfig.includeKinds.templates,
      generateTriggers = outOfScope,
      generateDeliveryPatterns = outOfScope,
      generateDashboards = exportConfig.includeKinds.dashboards,
      generateConfigurations = exportConfig.includeKinds.configurations,
      generateSettings = exportConfig.includeKinds.settings,
      generateRiskProfiles = exportConfig.includeKinds.riskProfiles,
      generateVariables = exportConfig.includeKinds.variables,
      generateNotificationSettings = exportConfig.includeKinds.notifications,
      generateCalendar = exportConfig.includeKinds.calendars
    )
  }

  def toGeneratorConfig(exportConfig: GlobalExportConfig, isAdminUser: Boolean): GeneratorConfig = {
    val name = if (exportConfig.nameFilter == null || exportConfig.nameFilter.isBlank) None else Some(exportConfig.nameFilter)
    GeneratorConfig(name, toSearchScope(exportConfig), toCisConfig(exportConfig), isAdminUser, exportConfig.excludeEntities.asScala.toSeq)
  }

  def validateAndNormalize(exportConfig: GlobalExportConfig): Unit = {
    val hasKinds = exportConfig.includeKinds.hasKinds()

    if (!hasKinds && exportConfig.includeEntities.isEmpty) {
      throw new IllegalArgumentException("You need to specify kinds or entities to export")
    }

    // If entities are specified only those kinds will be included
    // supporting a hybrid of specific items and all of other kinds requires
    // refactoring the generators to support multiple scopes
    if (!exportConfig.includeEntities.isEmpty) {
      val entities = exportConfig.includeEntities.asScala

      if (!entities.filterNot(Ids.isReleaseId).isEmpty) {
        throw new IllegalArgumentException("Only template ids can be specified in includeEntities")
      }

      if (!entities.filter(Ids.isInFolder).isEmpty) {
        throw new IllegalArgumentException("Only global ids can be specified")
      }

      exportConfig.includeKinds.templates = !entities.filter(Ids.isReleaseId).isEmpty
    }
  }
}

class GlobalExportKinds {
  @BeanProperty
  var applications: Boolean = false

  @BeanProperty
  var calendars: Boolean = false

  @BeanProperty
  var configurations: Boolean = false

  @BeanProperty
  var dashboards: Boolean = false

  @BeanProperty
  var environments: Boolean = false

  @BeanProperty
  var notifications: Boolean = false

  @BeanProperty
  var permissions: Boolean = false

  @BeanProperty
  var riskProfiles: Boolean = false

  @BeanProperty
  var roles: Boolean = false

  @BeanProperty
  var settings: Boolean = false

  @BeanProperty
  var templates: Boolean = false

  @BeanProperty
  var users: Boolean = false

  @BeanProperty
  var variables: Boolean = false

  def hasKinds(): Boolean = {
    applications ||
      calendars ||
      configurations ||
      dashboards ||
      environments ||
      notifications ||
      permissions ||
      riskProfiles ||
      roles ||
      settings ||
      templates ||
      users ||
      variables
  }
}

class GlobalExportConfig {
  @BeanProperty
  var nameFilter: String = _

  @BeanProperty
  var includeKinds: GlobalExportKinds = new GlobalExportKinds

  @BeanProperty
  var includeEntities: JList[String] = new JArrayList[String]

  @BeanProperty
  var excludeEntities: JList[String] = new JArrayList[String]
}
