package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.xlrelease.api.v1.filter.ReservationFilters
import com.xebialabs.xlrelease.api.v1.{EnvironmentApi, EnvironmentLabelApi, EnvironmentReservationApi, EnvironmentStageApi}
import com.xebialabs.xlrelease.ascode.service.GenerateService.GeneratorConfig
import com.xebialabs.xlrelease.ascode.utils.{DateUtils, Utils}
import com.xebialabs.xlrelease.ascode.yaml.model.{Environment, Label, Reservation}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.{EDIT_ENVIRONMENT, EDIT_RESERVATION}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.Date
import scala.jdk.CollectionConverters._

@Service
class EnvironmentsAsCodeService @Autowired()(stageApi: EnvironmentStageApi,
                                             labelApi: EnvironmentLabelApi,
                                             environmentApi: EnvironmentApi,
                                             environmentReservationApi: EnvironmentReservationApi,
                                             permissions: PermissionChecker) {

  def generateStages(generatorConfig: GeneratorConfig): List[String] = {
    if (generatorConfig.cisConfig.generateEnvironments &&
      !generatorConfig.searchScope.isInstanceOf[FolderSearch] &&
      permissions.hasGlobalPermission(EDIT_ENVIRONMENT)) {
      stageApi.search(null).asScala.toList
        .filter { stage =>
          generatorConfig.name.forall { n => Utils.stringLike(stage.getTitle, n) }
        }
        .map(_.getTitle)
    } else {
      List.empty
    }
  }

  def generateLabels(generatorConfig: GeneratorConfig): List[Label] = {
    if (generatorConfig.cisConfig.generateEnvironments &&
      !generatorConfig.searchScope.isInstanceOf[FolderSearch] &&
      permissions.hasGlobalPermission(EDIT_ENVIRONMENT)) {
      labelApi.search(null).asScala.toList
        .filter { label =>
          generatorConfig.name.forall { n => Utils.stringLike(label.getTitle, n) }
        }
        .map(v => Label(v.getTitle, v.getColor))
    } else {
      List.empty
    }
  }

  def generateEnvironments(generatorConfig: GeneratorConfig): List[Environment] = {
    if (generatorConfig.cisConfig.generateEnvironments &&
      !generatorConfig.searchScope.isInstanceOf[FolderSearch] &&
      permissions.hasGlobalPermission(EDIT_ENVIRONMENT)) {
      environmentApi.search(null).asScala.toList
        .filter { env =>
          generatorConfig.name.forall { n => Utils.stringLike(env.getTitle, n) }
        }
        .map(v => Environment(v.getTitle, Option(v.getDescription), v.getStage.getTitle, v.getLabels.asScala.map(_.getTitle).toList))
    } else {
      List.empty
    }
  }

  def generateEnvironmentReservations(generatorConfig: GeneratorConfig): List[Reservation] = {
    if (generatorConfig.cisConfig.generateEnvironments &&
      !generatorConfig.searchScope.isInstanceOf[FolderSearch] &&
      permissions.hasGlobalPermission(EDIT_RESERVATION)) {
      environmentReservationApi.search(new ReservationFilters(null, null, null, null, new Date(), null)).asScala
        .flatMap { case (_, reservations) =>
          reservations.asScala.toList
            .filter { envRes =>
              generatorConfig.name.forall { n => Utils.stringLike(envRes.getEnvironment.getTitle, n) }
            }
            .map { res =>
              Reservation(
                res.getEnvironment.getTitle,
                res.getApplications.asScala.map(_.getTitle).toList,
                DateUtils.toString(res.getStartDate),
                DateUtils.toString(res.getEndDate),
                res.getNote
              )
            }
        }.toList
    } else {
      List.empty
    }
  }
}
