package com.xebialabs.xlrelease.versioning.scheduler

import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration.QUARTZ_SCHEDULER_NAME
import com.xebialabs.xlrelease.quartz.events.SchedulerStartedEvent
import com.xebialabs.xlrelease.service.FolderService
import com.xebialabs.xlrelease.versioning.ascode.scm.FolderVersioningService
import com.xebialabs.xlrelease.versioning.ascode.upgrader.XLRelease233ContentFolderInitializer._
import grizzled.slf4j.Logging
import org.springframework.context.event
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

/**
 * This service schedules quartz job to auto import default workflows.
 * We have added upgrader, [[com.xebialabs.xlrelease.versioning.ascode.upgrader.XLRelease233ContentFolderInitializer]] to create default workflow folders.
 * We could not add this scheduling as there was no guarantee on upgrade execution path which resulted in error if Quartz was not initialized.
 * DO NOT expand this for all folders as this could have serious performance problem during system startup.
 * This is fine for workflows folder as we don't expect to have 500+ default workflows folder.
 */
@Service
class ScheduleDefaultContentFolderAutoApplyService(folderService: FolderService,
                                                   folderVersioningService: FolderVersioningService,
                                                   folderVersioningAutoApplyJobService: FolderVersioningAutoApplyJobService) extends Logging {

  @event.EventListener
  def onStartup(event: SchedulerStartedEvent): Unit = {
    if (null != event && null != event.schedulerName && event.schedulerName == QUARTZ_SCHEDULER_NAME) {
      Try {
        val maybeFolder = Try(folderService.findById(DEFAULT_CONTENT_FOLDER_ID)).toOption
        maybeFolder match {
          case Some(contentFolder) =>
            val subContentFolders = contentFolder.getChildren.asScala
            val workflowFolder = subContentFolders.find(_.getTitle == WORKFLOWS_FOLDER_NAME)

            workflowFolder.foreach { folder =>
              scheduleQuartzJobForFolder(folder.getId, folder.getTitle)
              folder.getChildren.asScala.foreach(child => scheduleQuartzJobForFolder(child.getId, child.getTitle))
            }
          case None =>
            logger.info(s"Not scheduling quartz jobs as default content folder '$DEFAULT_CONTENT_FOLDER_ID' does not exist")
        }
      } match {
        case Failure(ex) => logger.error(s"Failure while scheduling quartz jobs for default content folder auto import", ex)
        case Success(_) => logger.debug(s"Scheduled all required quartz job for default content folder auto import")
      }
    }
  }

  private def scheduleQuartzJobForFolder(folderId: String, folderName: String): Unit = {
    Try {
      val folderVersioningSettings = folderVersioningService.findSettings(folderId)
      folderVersioningSettings.foreach(folderVersioningAutoApplyJobService.handleAutoApplyGitVersion)
    } match {
      case Failure(exception) => logger.error(s"Failure while scheduling auto import for folder '$folderName'", exception)
      case Success(_) => logger.debug(s"Scheduled auto import for folder '$folderName'")
    }
  }
}
