package com.xebialabs.xlrelease.ascode.service.generatestrategy

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.service.GenerateService.CisConfig
import com.xebialabs.xlrelease.ascode.service._
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xltype.serialization.CiReference

case class CisWithRefs(cis: List[ConfigurationItem] = Nil, references: List[CiReference] = Nil) {
  def merge(another: CisWithRefs): CisWithRefs = CisWithRefs(cis ::: another.cis, references ::: another.references)
}

case class CiGenerateConfig(ciTitle: Option[String] = None,
                            searchScope: SearchScope = AllSearch,
                            folders: Seq[Folder] = Seq.empty,
                            generateConfig: CisConfig,
                            excludedEntities: Seq[String] = Seq.empty,
                            isAdminUser: Boolean = false,
                            folderGeneratePermissions: Map[String, Boolean] = Map.empty)

trait GenerateStrategy[T <: ConfigurationItem] extends PartialFunction[CiGenerateConfig, Map[Option[Folder], List[ConfigurationItem]]] {
  /**
   * This order will be used to sort generate strategies
   */
  val generateOrder: Double

  override def apply(config: CiGenerateConfig): Map[Option[Folder], List[ConfigurationItem]] = {
    val folderToCis = config.searchScope match {
      case ExactSearch(ids) => generateByIds(ids, config)
      case GlobalSearch => generateGlobal(config)
      case FolderSearch(path, id, includeSubFolders) => generateFolder(config)
      case AllSearch => generateGlobal(config) ++ generateFolder(config)
    }
    var filtered = config.ciTitle.fold(folderToCis)(filter(folderToCis, _))
    if (config.excludedEntities.nonEmpty) {
      filtered = filtered.map { case (maybeFolder, cis) =>
        maybeFolder -> cis.filterNot { ci =>
          config.excludedEntities.contains(ci.getId)
        }
      }
    }
    filtered
  }

  protected def generateByIds(ids: List[String], config: CiGenerateConfig): Map[Option[Folder], List[T]] = {
    Map.empty
  }

  protected def generateGlobal(config: CiGenerateConfig): Map[Option[Folder], List[T]] = {
    Map.empty
  }

  protected def generateFolder(config: CiGenerateConfig): Map[Option[Folder], List[T]] = {
    Map.empty
  }

  protected def filter(results: Map[Option[Folder], List[T]], ciTitle: String): Map[Option[Folder], List[T]] = {
    results.map { case (maybeFolder, cis) =>
      maybeFolder -> cis.filter { ci =>
        Utils.stringLike(ci.getProperty[String]("title"), ciTitle)
      }
    }
  }
}

trait GenericGenerateStrategy extends GenerateStrategy[ConfigurationItem]
