package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.yaml.dto.AsCodeResponse.ChangedIds
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.xlrelease.api.v1.UserApi
import com.xebialabs.xlrelease.api.v1.forms.UserAccount
import com.xebialabs.xlrelease.ascode.service.GenerateService.GeneratorConfig
import com.xebialabs.xlrelease.ascode.service.UserUtils._
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.yaml.model.user.XLRUser
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.stream.Collectors
import jakarta.ws.rs.NotFoundException
import scala.jdk.CollectionConverters._

@Service
class UsersAsCodeService @Autowired()(
                                       userApi: UserApi
                                     ) extends Logging {
  def generateUsers(generatorConfig: GeneratorConfig): List[XLRUser] = {
    if (generatorConfig.cisConfig.generateUsers && !generatorConfig.searchScope.isInstanceOf[FolderSearch]) {
      val users = userApi.findUsers(null, null, null, null, null, null, null, null)
      users
        .stream()
        .filter(!_.isExternal)
        .filter { userAccount =>
          generatorConfig.name.forall { n => Utils.stringLike(userAccount.getUsername, n) }
        }
        .map[XLRUser](userAccountToUser)
        .collect(Collectors.toList[XLRUser])
        .asScala
        .toList
    } else {
      List.empty
    }
  }

  def handleUser(user: XLRUser): ChangedIds = {
    val transformedUser = userToUserAccount(user)
    val ids = USER.ids

    try {
      val storedUser = userApi.getUser(user.username)
      transformedUser.setPassword(user.password)

      val updatedUser = userApi.updateUser(transformedUser)
      userApi.updatePassword(user.username, storedUser.getPassword, user.password)
      ids.withUpdated(updatedUser.getUsername)
    } catch {
      case _: NotFoundException =>
        val createdUser = userApi.createUser(transformedUser)
        ids.withCreated(createdUser.getUsername)
    }
  }
}

object UserUtils {
  def userToUserAccount(user: XLRUser): UserAccount = {
    val userAccount = new UserAccount

    userAccount.setUsername(user.username)
    userAccount.setPassword(user.password)
    user.email.foreach(userAccount.setEmail)
    user.name.foreach(userAccount.setFullName)
    userAccount.setLoginAllowed(user.enabled)
    user.dateFormat.foreach(userAccount.setDateFormat)
    user.timeFormat.foreach(userAccount.setTimeFormat)
    user.firstDayOfWeek.foreach(day => userAccount.setFirstDayOfWeek(dayOfWeekToNumber(day)))

    userAccount
  }

  def userAccountToUser(userAccount: UserAccount): XLRUser = {
    XLRUser(
      userAccount.getUsername,
      userAccount.getPassword,
      Option(userAccount.getFullName).filterNot(_.isEmpty),
      Option(userAccount.getEmail).filterNot(_.isEmpty),
      userAccount.isLoginAllowed,
      Option(userAccount.getDateFormat),
      Option(userAccount.getTimeFormat),
      Option(numberToDayOfWeek(userAccount.getFirstDayOfWeek))
    )
  }

  def dayOfWeekToNumber(day: String): Int = {
    day match {
      case "Sun" => 0
      case "Mon" => 1
      case _ => -1
    }
  }

  def numberToDayOfWeek(number: Integer): String = {
    Int.unbox(number) match {
      case 0 => "Sun"
      case 1 => "Mon"
      case _ => "Default"
    }
  }
}
