package com.xebialabs.xlrelease.api.internal

import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.scm.data.{VersionInfo, VersionsView}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.{APPLY_FOLDER_CHANGES, EDIT_FOLDER_VERSIONS, GENERATE_FOLDER_CONFIGURATION, VIEW_FOLDER_VERSIONS}
import com.xebialabs.xlrelease.versioning.ascode.actors.FolderVersioningActorService
import com.xebialabs.xlrelease.versioning.ascode.form.{CreateVersionForm, ValidationReport, ValidationReportMessage}
import com.xebialabs.xlrelease.versioning.ascode.scm.{FolderVersioningPreviewService, FolderVersioningService, VersioningStyle}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import jakarta.ws.rs._
import jakarta.ws.rs.core.MediaType
import scala.jdk.CollectionConverters._

@Path("api/v1/folders/versioning")
@Produces(Array(MediaType.APPLICATION_JSON))
@Consumes(Array(MediaType.APPLICATION_JSON))
@Controller
class FolderVersioningResource @Autowired()(permissionChecker: PermissionChecker,
                                            folderVersioningService: FolderVersioningService,
                                            previewService: FolderVersioningPreviewService,
                                            folderVersioningActorService: FolderVersioningActorService) extends Logging {

  @GET
  @Path("/{folderId:.*Folder[^/-]*}/")
  def getSettings(@PathParam("folderId") folderId: String): FolderVersioningSettings = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.findSettings(folderId).orNull
  }

  @PUT
  @Path("/{folderId:.*Folder[^/-]*}/")
  def updateSettings(@PathParam("folderId") folderId: String, config: FolderVersioningSettings): FolderVersioningSettings = {
    permissionChecker.check(EDIT_FOLDER_VERSIONS, folderId)
    config.setFolderId(folderId)
    folderVersioningActorService.createOrUpdateSettings(config)
  }

  @DELETE
  @Path("/{folderId:.*Folder[^/-]*}/")
  def deleteSettings(@PathParam("folderId") folderId: String): Unit = {
    permissionChecker.check(ADMIN)
    folderVersioningActorService.cleanLocalRepo(folderId, true)
    folderVersioningService.deleteSettings(folderId)
  }

  @GET
  @Path("/{folderId:.*Folder[^/-]*}/versions")
  def getVersions(@PathParam("folderId") folderId: String, @DefaultValue("false") @QueryParam("fetchChanges") fetchChanges: Boolean): VersionsView = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    val (lastFetched, versions) = if (fetchChanges) {
      folderVersioningActorService.fetchChanges(folderId)
    } else {
      folderVersioningService.getVersions(folderId)
    }
    new VersionsView(lastFetched, versions.asJava)
  }

  @POST
  @Path("/{folderId:.*Folder[^/-]*}/versions")
  def createVersion(@PathParam("folderId") folderId: String, versionForm: CreateVersionForm): VersionInfo = {
    permissionChecker.check(GENERATE_FOLDER_CONFIGURATION, folderId)
    folderVersioningActorService.createVersion(folderId, versionForm.name, versionForm.description)
  }

  @POST
  @Path("/{folderId:.*Folder[^/-]*}/versions/{version:.*}/apply")
  def applyVersion(@PathParam("folderId") folderId: String, @PathParam("version") version: String): Unit = {
    // TODO: which permission to check here?
    permissionChecker.checkAny(folderId, APPLY_FOLDER_CHANGES)
    folderVersioningService.applyVersion(folderId, version)
  }

  @GET
  @Produces(Array("text/vnd.yaml"))
  @Path("/{folderId:.*Folder[^/-]*}/preview")
  def previewCurrent(@PathParam("folderId") folderId: String, @QueryParam("filename") filename: String): String = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    previewService.generatePreview(folderId, None, filename)
  }

  @GET
  @Produces(Array("text/vnd.yaml"))
  @Path("/{folderId:.*Folder[^/-]*}/versions/{version:.*}/preview")
  def previewVersion(@PathParam("folderId") folderId: String, @PathParam("version") version: String, @QueryParam("filename") filename: String): String = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    previewService.generatePreview(folderId, Some(version), filename)
  }

  @GET
  @Path("/{folderId:.*Folder[^/-]*}/versions/{version:.*}/getVersionedFileNames")
  def getVersionedFileNames(@PathParam("folderId") folderId: String, @PathParam("version") version: String): List[String] = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.getVersionedFileNames(folderId, version)
  }

  @GET
  @Path("/{folderId:.*Folder[^/-]*}/getVersionableFileNames")
  def getCurrentVersionableFileNames(@PathParam("folderId") folderId: String): List[String] = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.getVersionableFileNames(folderId)
  }

  @DELETE
  @Path("/{folderId:.*Folder[^/-]*}/delete-local-repo")
  def deleteLocalRepo(@PathParam("folderId") folderId: String, @DefaultValue("false") @QueryParam("clusterWide") clusterWide: Boolean): Unit = {
    permissionChecker.check(ADMIN)
    folderVersioningActorService.cleanLocalRepo(folderId, clusterWide)
  }

  @PUT
  @Path("/{folderId:.*Folder[^/-]*}/reset-local-repo")
  def resetLocalRepo(@PathParam("folderId") folderId: String, @DefaultValue("false") @QueryParam("clusterWide") clusterWide: Boolean): Unit = {
    permissionChecker.check(ADMIN)
    val config = folderVersioningService.getSettings(folderId)
    folderVersioningActorService.resetLocalRepo(config, clusterWide)
  }

  @GET
  @Path("/{folderId:.*Folder[^/-]*}/validate")
  def validateCurrent(@PathParam("folderId") folderId: String): ValidationReport = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.validate(folderId)
  }

  @GET
  @Path("/{folderId:.*Folder[^/-]*}/validation-messages")
  def getValidationMessages(@PathParam("folderId") folderId: String): ValidationReport = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.getValidationMessages(folderId)
  }

  @PUT
  @Path("/{folderId:.*Folder[^/-]*}/validation-messages")
  def setValidationMessages(@PathParam("folderId") folderId: String, validationReport: ValidationReport): Unit = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.setValidationMessages(folderId, validationReport.warnings.asScala.toSeq)
  }

  @GET
  @Path("/{folderId:.*Folder[^/-]*}/message/{messageId:.*}/")
  def getValidationMessage(@PathParam("folderId") folderId: String, @PathParam("messageId") messageId: String): ValidationReportMessage = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.getValidationMessage(messageId)
  }

  @PUT
  @Path("/{folderId:.*Folder[^/-]*}/message/{messageId:.*}/update")
  def setValidationMessagesStatus(@PathParam("folderId") folderId: String, @PathParam("messageId") messageId: String, validationMessage: ValidationReportMessage): Unit = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.updateValidationMessageStatus(messageId, validationMessage)
  }

  @DELETE
  @Path("/{folderId:.*Folder[^/-]*}/clear-secrets")
  def clearSecrets(@PathParam("folderId") folderId: String): Unit = {
    permissionChecker.check(ADMIN)
    folderVersioningActorService.clearSecrets(folderId)
  }

  @GET
  @Path("/styles")
  def getVersioningStyles: List[VersioningStyle] = {
    folderVersioningService.getVersioningStyles
  }
}
