package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.Utils.splitStringByPathSeparator
import com.xebialabs.ascode.utils.{Utils => CommonUtils}
import com.xebialabs.xlrelease.api.v1.FolderApi
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.utils.Utils.getFolderPathFromCiPath
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings.ENCODED_SEPARATOR
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.{ROOT_FOLDER_ID, SEPARATOR}
import com.xebialabs.xlrelease.repository.sql.persistence.FolderPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow.Root
import com.xebialabs.xlrelease.repository.{FolderRepository, Ids}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.{CREATE_TOP_LEVEL_FOLDER, EDIT_FOLDER, GENERATE_FOLDER_CONFIGURATION, VIEW_FOLDER}
import com.xebialabs.xlrelease.service.FolderService
import org.apache.commons.lang.StringEscapeUtils.unescapeJava
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._
import scala.util.Try

@Service
class FolderAsCodeService @Autowired()(folderApi: FolderApi,
                                       folderService: FolderService,
                                       folderRepository: FolderRepository,
                                       folderPersistence: FolderPersistence,
                                       permissions: PermissionChecker,
                                       eventBus: XLReleaseEventBus) {

  def searchParentFolder(ciPath: String, metadata: Map[String, String]): Option[Folder] = {
    val homeFolder = metadata.getOrElse(MetadataFields.HOMEFOLDER.toString, SEPARATOR)
    val folderPath = Utils.buildFolderPath(homeFolder, getFolderPathFromCiPath(ciPath).getOrElse(""))
    searchFolder(folderPath)
  }

  def searchFolderById(releaseId: String): Option[Folder] = {
    if (Ids.isInFolder(releaseId)) {
      val folderId = Ids.findFolderId(releaseId)
      Some(folderApi.getFolder(folderId))
    } else {
      None
    }
  }

  def findFolderTitleById(folderId: String) =
    Try(folderService.getTitle(folderId)).getOrElse("")


  def searchFolder(ciPath: String, includeSubFolders: Boolean = true, failOnNotFound: Boolean = true): Option[Folder] = {
    val folderPath = splitStringByPathSeparator(ciPath).map(unescapeJava)
    searchFolder(folderPath, includeSubFolders, failOnNotFound)
  }

  private def searchFolder(ciPath: Array[String], includeSubFolders: Boolean, failOnNotFound: Boolean): Option[Folder] = {
    if (ciPath.headOption.contains(SEPARATOR)) {
      None
    } else {
      val folder = ciPath
        .headOption
        .flatMap(folderPersistence.findSubFolderByTitle(_, ROOT_FOLDER_ID))
        .map(_.uid)
        .map(folderPersistence.findByUid(_, depth = if (includeSubFolders) Int.MaxValue else (ciPath.length - 1)))
        .map(_.toFolder)
        .flatMap(foundFolder => ciPath.tail.foldLeft(Option(foundFolder)) { (acc, folderName) =>
          acc.flatMap(_.getChildren.asScala.find(_.getTitle == folderName))
        })

      folder.foreach(folder => checkFolderPermission(folder.getId))
      if (folder.isEmpty) {
        if (failOnNotFound) {
          throw new AsCodeException(s"Folder with path [${ciPath.mkString(SEPARATOR)}] was not found")
        } else {
          None
        }
      }
      folder
    }
  }

  def findSubFolderRowByTitle(title: String, parentId: String): Option[FolderRow] = {
    folderPersistence.findSubFolderByTitle(title, parentId)
  }

  private def findFolderByTitle(title: String, parentId: String): Option[Folder] = {
    folderPersistence.findSubFolderByTitle(title, parentId).map { row =>
      val folder = row.asFolder
      permissions.check(VIEW_FOLDER, folder.getId)
      folder
    }
  }

  def createFolder(title: String, parentId: String): Folder = {
    val folder = new Folder
    folder.setId(null)
    folder.setTitle(unescapeJava(title))
    folderApi.addFolder(parentId, folder)
  }

  case class FolderAndPostCommitActions(folder: Folder, postCommitActions: Seq[PostCommitAction] = Seq.empty)

  def createFolderInTransaction(title: String, parentId: String): FolderAndPostCommitActions = {
    val folder = new Folder
    folder.setId(null)
    folder.setTitle(unescapeJava(title).replaceAll(ENCODED_SEPARATOR, Ids.SEPARATOR))
    if (parentId.equals(ROOT_FOLDER_ID)) {
      permissions.check(CREATE_TOP_LEVEL_FOLDER)
    } else {
      permissions.check(EDIT_FOLDER, parentId)
    }

    val result = folderService.createWithoutPublishing(parentId, folder, true)

    FolderAndPostCommitActions(
      result.folder,
      result.events.map(event => () => eventBus.publish(event))
    )
  }

  def createFolderHierarchy(folderPath: String): Option[Folder] = {
    val folderNames = splitStringByPathSeparator(folderPath).map(name => name.replace(ENCODED_SEPARATOR, Ids.SEPARATOR))
    val leafFolder = folderNames.foldLeft(Root.asFolder) { case (parentFolder, folderName) =>
      findFolderByTitle(folderName, parentFolder.getId).getOrElse(createFolder(folderName, parentFolder.getId))
    }
    Some(leafFolder)
  }

  private case class FolderPathContainer(path: String, folder: Folder)

  def generateAbsolutePath(id: String, title: String): String = {
    val escapedTitle = CommonUtils.escapePath(title)

    val folderId = Ids.getParentId(id)
    if (ROOT_FOLDER_ID.equals(folderId)) {
      escapedTitle
    } else {
      val folderPath = folderRepository.getPath(folderId)
      Utils.joinPaths(folderPath.slice(1, folderPath.length) :+ escapedTitle)
    }
  }

  def getFolderPath(folderId: String): String = {
    Utils.joinPaths(folderRepository.getPath(folderId).tail.map(p => p.replaceAll(Ids.SEPARATOR, ENCODED_SEPARATOR)))
  }

  def getFolderRow(folderId: String): FolderRow = {
    folderPersistence.findById(folderId, depth = 0).value
  }

  private def checkFolderPermission(folderId: String): Unit = {
    if (!permissions.hasPermission(GENERATE_FOLDER_CONFIGURATION, folderId)) {
      permissions.check(VIEW_FOLDER, folderId)
    }
  }
}

object FolderAsCodeService {
  def absolutePath(referencePath: String, home: Option[String]): String = {

    Option(referencePath) match {
      case Some(value) => home.map { path =>
        if (referencePath == ".")
          path
        else if (referencePath.startsWith("./") && path != None)
          s"$path${referencePath.substring(1)}"
        else
          referencePath
      }.getOrElse(referencePath)
      case None => null
    }
  }

}
