package com.xebialabs.xlrelease.versioning.ascode.scm.strategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service.{FolderAsCodeService, FolderSearch, GenerateService}
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings._
import com.xebialabs.xlrelease.domain.{Release, Trigger}
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.scm.connector.ScmException
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.service.FolderService
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.MixedMultiFileStrategy.{MULTIPLE_FILES_PER_CI_TYPE, MULTIPLE_FILES_PER_CI_TYPE_VALUE}
import org.springframework.stereotype.Component

import java.text.Normalizer

object MixedMultiFileStrategy {
  val MULTIPLE_FILES_PER_CI_TYPE = "multipleFilesPerCIType"
  val MULTIPLE_FILES_PER_CI_TYPE_VALUE = "Multiple files - one for each entity"
}

/**
 * Some entity types are grouped within one file, some are split across multiple files.
 *
 * One file per entity (every entity of a below listed type will end up in it's own file):
 * - templates
 * - patterns
 * - triggers
 * - dashboards
 *
 * One file per type (all entities of a type go into one file):
 * - variables
 * - notifications
 * - configurations
 * - permissions
 * - applications
 * - environments
 */
@Component
class MixedMultiFileStrategy(permissionsChecker: PermissionChecker,
                             generateService: GenerateService,
                             folderService: FolderService,
                             folderAsCodeService: FolderAsCodeService) extends BaseVersioningStrategy {

  override def name: String = MULTIPLE_FILES_PER_CI_TYPE

  override def value: String = MULTIPLE_FILES_PER_CI_TYPE_VALUE

  def performGenerate(scope: FolderSearch, rootPath: String, settings: FolderVersioningSettings): Map[String, List[Definition]] = {
    val ciConfigs = settings.splitAndConvert()
    val excludeIds = Seq(settings.gitConnection.getId)
    ciConfigs.flatMap(performGenerateInternal(_, scope, rootPath, excludeIds)).toMap
  }

  def performGenerateInternal(ciConfig: CisConfig, scope: FolderSearch, rootPath: String, excludeIds: Seq[String]): Map[String, List[Definition]] = {
    try {
      val generatorConfig = GeneratorConfig(None, scope, ciConfig, permissionsChecker.hasGlobalPermission(ADMIN), excludeIds)
      val definitions = generateService.generate(generatorConfig)

      ciConfig match {
        case c: CisConfig if c.isFilePerEntity => extractEntities(definitions, scope.path, rootPath, c)
        case c: CisConfig if c.isFilePerType => Map(getDefinitionsPath(scope.path, resolveFileName(ciConfig)) -> definitions.map(_.adjustMetadata(rootPath)))
        case _ => Map()
      }
    } catch {
      case e: AsCodeException =>
        e.getMessage match {
          case GenerateService.EMPTY_DEFINITION_ERROR => Map()
          case _ => throw ScmException(e.getMessage, e.getCause)
        }
    }
  }

  private def extractEntities(definitions: List[Definition],
                              entitiesPath: String,
                              rootPath: String,
                              cisConfig: CisConfig): Map[String, List[Definition]] = {
    val definitionsWithFileNames = definitions.flatMap { definition =>
      val ciSpec = definition.spec.asInstanceOf[CiSpec]
      ciSpec.cis.map { ci =>
        val fileName = getDefinitionsPath(entitiesPath, resolveFileName(cisConfig, ci.getTitle))
        val updatedDef = definition.adjustMetadata(rootPath)
        (fileName, List(updatedDef.copy(spec = ciSpec.copy(cis = List(ci)))))
      }
    }

    definitionsWithFileNames.foldLeft(Map.empty[String, List[Definition]]) { case (acc, (fileName, definition)) =>
      val newFileName = if (acc.contains(fileName)) {
        val (base, ext) = fileName.splitAt(fileName.lastIndexOf('.'))
        LazyList.from(1).map(i => s"${base}_$i$ext").find(!acc.contains(_)).get
      } else {
        fileName
      }
      acc + (newFileName -> definition)
    }
  }

  private def resolveFileName(ciConfig: CisConfig, title: String) = {
    val prefix = ciConfig match {
      case c: CisConfig if c.generateTemplates => TEMPLATES_VERSIONING_YAML_FILENAME_PREFIX
      case c: CisConfig if c.generateWorkflows => WORKFLOWS_VERSIONING_YAML_FILENAME_PREFIX
      case c: CisConfig if c.generateDeliveryPatterns => PATTERNS_VERSIONING_YAML_FILENAME_PREFIX
      case c: CisConfig if c.generateTriggers => TRIGGERS_VERSIONING_YAML_FILENAME_PREFIX
      case c: CisConfig if c.generateDashboards => DASHBOARDS_VERSIONING_YAML_FILENAME_PREFIX
      case _ => ""
    }

    s"$prefix${sanitizeTitle(title)}.yaml"
  }

  private def sanitizeTitle(title: String): String = {
    val sanitizedTitle = Normalizer.normalize(title, Normalizer.Form.NFD)
      .replaceAll("""[^\p{ASCII}]""", "")
      .toLowerCase()
      .replaceAll("""[^a-z0-9\s-]""", "_")
      .replaceAll("""\s+""", "_")
      .replaceAll("""_+""", "_")
      .replaceAll("""_$""", "")
    sanitizedTitle
  }

  private def resolveFileName(ciConfig: CisConfig) = {
    ciConfig match {
      case c: CisConfig if c.generateConfigurations => CONNECTIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateVariables => VARIABLES_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generatePermissions => PERMISSIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateNotificationSettings => NOTIFICATIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateApplications => APPLICATIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateEnvironments => ENVIRONMENTS_VERSIONING_YAML_FILENAME
      case _ => ""
    }
  }

  implicit class CisConfigOps(c: CisConfig) {
    def isFilePerType: Boolean = {
      c.generateVariables ||
        c.generateConfigurations ||
        c.generateNotificationSettings ||
        c.generatePermissions ||
        c.generateApplications ||
        c.generateEnvironments
    }

    def isFilePerEntity: Boolean = {
      c.generateTemplates || c.generateWorkflows || c.generateDeliveryPatterns || c.generateTriggers || c.generateDashboards
    }
  }

  implicit class ConfigurationItemOps(ci: ConfigurationItem) {
    def getTitle: String = {
      ci match {
        case template: Release => template.getTitle
        case dashboard: Dashboard => dashboard.getTitle
        case trigger: Trigger => trigger.getTitle
        case pattern: Delivery => pattern.getTitle
        case _ => ci.getName
      }
    }
  }

  override def order: Integer = 1

  override def getFolderService: FolderService = folderService

  override def getFolderAsCodeService: FolderAsCodeService = folderAsCodeService

  override def getPermissions: PermissionChecker = permissionsChecker

  override def isMultiFile: Boolean = true

}