package com.xebialabs.xlrelease.versioning.ascode.scm

import com.xebialabs.ascode.exception.{AsCodeException, DocumentValidationException}
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.api.v1.views.VersionInfo
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.service.GenerateService.EMPTY_DEFINITION_ERROR
import com.xebialabs.xlrelease.ascode.service._
import com.xebialabs.xlrelease.ascode.utils.{StaticVariables, TemplateUtils}
import com.xebialabs.xlrelease.ascode.yaml.writer.XLRDefinitionWriter
import com.xebialabs.xlrelease.domain.configuration.connector.{GitProvider, Repository}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.domain.versioning.ascode._
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettingsUtil._
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.{FolderInfo, ValidationReport, ValidationReportMessage}
import com.xebialabs.xlrelease.domain.{Release, Trigger}
import com.xebialabs.xlrelease.repository.{Ids, TriggerRepository}
import com.xebialabs.xlrelease.scm.connector.JGitConnector._
import com.xebialabs.xlrelease.scm.connector._
import com.xebialabs.xlrelease.scm.data.{GitTag, ValidatedCommitInfo}
import com.xebialabs.xlrelease.service._
import com.xebialabs.xlrelease.triggers.service.TriggerService
import com.xebialabs.xlrelease.versioning.ascode.VersioningUtils.getEffectiveTagPrefix
import com.xebialabs.xlrelease.versioning.ascode.form.CreateVersionResult
import com.xebialabs.xlrelease.versioning.ascode.repository.FolderValidationMessageRepository
import com.xebialabs.xlrelease.versioning.ascode.scm.FolderVersioningService.wrapExceptions
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.AsCodeJGitConnectorInitializer
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.VersioningStrategyResolver
import com.xebialabs.xlrelease.versioning.ascode.{ExportValidator, ImportValidator}
import grizzled.slf4j.Logging
import org.eclipse.jgit.api.errors.{GitAPIException, JGitInternalException}
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Service
import org.springframework.transaction.support.TransactionTemplate

import java.io.ByteArrayOutputStream
import java.util.Date
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.util.control.Breaks.{break, breakable}
import scala.util.control.Exception
import scala.util.control.Exception.catchingPromiscuously
import scala.util.{Try, Using}

object FolderVersioningService {

  // wrapping all exceptions related to SCM operations
  lazy val wrapExceptions: Exception.Catch[Nothing] = catchingPromiscuously(classOf[AsCodeException], classOf[GitAPIException], classOf[JGitInternalException])
    .withApply {
      case validationEx: DocumentValidationException =>
        throw ScmException(s"Validation error on field: ${validationEx.error.field}, Problem: ${validationEx.error.problem}")
      case e =>
        throw ScmException(s"Error performing folder Git versioning operation: ${e.getMessage}", e.getCause)
    }
}

@Service
class FolderVersioningService @Autowired()(userProfileService: UserProfileService,
                                           folderVersioningConfigService: FolderVersioningConfigService,
                                           connectorInitializer: AsCodeJGitConnectorInitializer,
                                           folderAsCodeService: FolderAsCodeService,
                                           releaseService: ReleaseService,
                                           val folderService: FolderService,
                                           @Qualifier("xlrRepositoryTransactionTemplate")
                                           transactionTemplate: TransactionTemplate,
                                           importService: ImportService,
                                           orphanDeletionService: OrphanDeletionService,
                                           folderValidationMessageRepository: FolderValidationMessageRepository,
                                           triggerRepository: TriggerRepository,
                                           triggerService: TriggerService,
                                           versioningStrategyResolver: VersioningStrategyResolver,
                                           definitionsGenerator: DefinitionsGenerator,
                                           definitionCrawler: DefinitionCrawler,
                                           importCrawler: ImportCrawler
                                          ) extends Logging {

  def findSettings(folderId: String): Option[FolderVersioningSettings] = wrapExceptions {
    folderVersioningConfigService.findSettings(folderId)
  }

  def findSettingsById(configId: String): FolderVersioningSettings = wrapExceptions {
    folderVersioningConfigService.findSettingsByConfigId(configId)
  }

  def getSettings(folderId: String): FolderVersioningSettings = wrapExceptions {
    findSettings(folderId)
      .getOrElse(throw ScmException("Folder Git versioning configuration is not defined for this folder."))
  }

  def createOrUpdateSettings(newConfig: FolderVersioningSettings, remoteValidations: Boolean = true): FolderVersioningSettings = wrapExceptions {
    validateSettings(newConfig, remoteValidations)
    folderVersioningConfigService.createOrUpdateSettings(newConfig)
  }

  def deleteSettings(folderId: String): Unit = wrapExceptions {
    folderVersioningConfigService.deleteSettings(folderId)
  }

  def getVersions(folderId: String): (Date, Seq[VersionInfo]) = wrapExceptions {
    val config = getSettings(folderId)
    Using.resource(connectorInitializer.init(config)) { scmConnector =>
      val versions = scmConnector.listVersions(Some(config))
      populateCommitUrls(config.getGitConnection, versions)
      config.getLastFetched -> versions.sortBy(_.getCommitTime)(Ordering[Date].reverse)
    }
  }

  def fetchChanges(folderId: String): (Date, Seq[VersionInfo]) = wrapExceptions {
    val config = getSettings(folderId)
    Using.resource(connectorInitializer.init(config)) { scmConnector =>
      val versions = scmConnector.pullAndListVersions(Some(config))
      config.setLastFetched(new Date())
      // don't fire update event on this update
      folderVersioningConfigService.updateSettingsDirectly(config)
      populateCommitUrls(config.getGitConnection, versions)
      config.getLastFetched -> versions.sortBy(_.getCommitTime)(Ordering[Date].reverse)
    }
  }

  def getAllBranches(configId: String): Seq[GitBranch] = wrapExceptions {
    val config = findSettingsById(configId)
    Using.resource(connectorInitializer.init(config)) { scmConnector =>
      scmConnector.getAllBranches.map(branch => GitBranch(branch, branch)).sortBy(_.title)
    }
  }

  def createVersion(folderId: String, version: String, description: String): CreateVersionResult = wrapExceptions {
    val maxBytes = 255
    val byteLength = description.getBytes("UTF-8").length
    if (byteLength > maxBytes) {
      throw new IllegalArgumentException(s"Description exceeds $maxBytes bytes (was $byteLength bytes)")
    }

    val config = getSettings(folderId)
    val latestTag = getVersions(folderId)._2.headOption.map(_.getName)
    val filesFromPreviousVersion = latestTag match {
      case Some(tag) => getVersionedFileNames(folderId, tag)
      case None => List()
    }

    Using.resource(connectorInitializer.init(config)) { scmConnector =>

      val folder = versioningStrategyResolver.resolve(config.getVersioningStyle).generateFolder(folderId, config)
      val tag = GitTag(generateTagName(getEffectiveTagPrefix(config.getScmPath), version))
      val commitInfo = ValidatedCommitInfo.create(version, description, tag).get
      val userProfile = userProfileService.findByUsername(getAuthenticatedUserName)
      val filesFromCurrentVersion = folder.blob.filesToAdd.map(_.absolutePath)
      val filesToRemove = filesFromPreviousVersion.map(getDefinitionsPath(config.getScmPath, _)).filterNot(filesFromCurrentVersion.contains(_))

      scmConnector.createNewVersion(ScmBlobs(folder.blob.filesToAdd, filesToRemove), commitInfo, userProfile).get

      val createVersionResult = CreateVersionResult
        .versionInfoToCreateVersionResult(scmConnector.getVersion(commitInfo.tag.refName))

      // put secrets in generated placeholders
      config.getSecrets.putAll(folder.secrets.asJava)
      // set applied version
      markAsApplied(createVersionResult.getName, config)
      folderVersioningConfigService.updateSettingsDirectly(config)

      val scmData = getScmTraceabilityData(config, createVersionResult)
      applyScmDataToDefinitions(folder.definitions, scmData)

      setValidationMessages(folderId, Seq.empty)

      createVersionResult
    }
  }

  def validate(folderId: String): ValidationReport = {
    val config = getSettings(folderId)
    val definitions = definitionsGenerator.generate(folderId, config)
    val folderInfo = new FolderInfo(folderId, "", folderAsCodeService.getFolderPath(folderId))
    ValidationReport.messagesToReport(ExportValidator.validate(definitions, folderInfo, folderService, releaseService, fromApply = false).asJava)
  }

  // Sorts definitions so that any template referenced by a CRT come first
  def sortByCRTReferences(defs: Seq[Definition]): Seq[Definition] = {
    def getRelease(d: Definition): Option[Release] =
      d.spec match {
        case ci: CiSpec =>
          ci.cis.headOption.collect { case r: Release => r }
        case _ => None
      }

    def releaseTitle(d: Definition): Option[String] =
      d.spec match {
        case ci: CiSpec =>
          ci.cis.headOption.collect { case r: Release => r.getTitle }
        case _ => None
      }

    val crtIds = ListBuffer.empty[String]
    defs.foreach { d =>
      if (d.kind == StaticVariables.XLR_TEMPLATE_KIND) {
        getRelease(d).foreach { release =>
          TemplateUtils.forEachCreateReleaseTask(release) { task =>
            Option(TemplateUtils.getTemplateTitleByTemplateId(task.getTemplateId)).foreach(crtIds += _)
          }
        }
      }
    }

    defs.sortBy(d => !releaseTitle(d).exists(crtIds.contains))
  }

  def applyVersion(folderId: String, version: String): ValidationReport = wrapExceptions {
    val settings = getSettings(folderId)
    val includes = ImportIncludes(
      templates = settings.getExportTemplates,
      workflows = settings.getExportWorkflows,
      triggers = settings.getExportTriggers,
      patterns = settings.getExportPatterns,
      dashboards = settings.getExportDashboards,
      configurations = settings.getExportConfiguration,
      variables = settings.getExportVariables,
      notifications = settings.getExportNotifications,
      permissions = settings.getExportSecurity,
      applications = settings.getExportApplications,
      environments = settings.getExportEnvironments
    )

    Using.resource(connectorInitializer.init(settings)) { scmConnector =>
      val versionInfo = scmConnector.getVersion(version)
      val scmData = getScmTraceabilityData(settings, versionInfo)

      val allDefinitions = sortByCRTReferences(definitionCrawler.findAllDefinitions(settings.getScmPath, scmConnector, version))

      val fileResolver = new ScmFileResolver(scmConnector, settings, version)
      ArtifactResolver(fileResolver).resolveArtifacts(allDefinitions: _*)

      settings.getSecrets.putAll(getFolderSecrets(folderId, settings).asJava)

      val folderPath = folderAsCodeService.getFolderPath(folderId).replaceAll(Ids.SEPARATOR, ENCODED_SEPARATOR)
      val folderInfo = new FolderInfo(folderId, "", folderPath)
      val currentSecrets = settings.getSecrets.asScala.toMap
      val validator = new ImportValidator(currentSecrets, allDefinitions, folderInfo, folderService, releaseService, fromApply = true)

      val result = applyDefinitionsInTransaction(includes, folderId, allDefinitions, Some(scmData), Some(settings.getGitConnection), Some(validator))

      markAsApplied(version, settings)
      folderVersioningConfigService.updateSettingsDirectly(settings)

      val validationReport = ValidationReport.messagesToReport(result.validationMessages.asJava)
      setValidationMessages(folderId, validationReport.getWarnings.asScala.toSeq)
      ValidationReport.createWarningReport(getValidationMessages(folderId).getWarnings)
    }
  }

  def getVersionedFileNames(folderId: String, version: String): List[String] = wrapExceptions {
    val settings = getSettings(folderId)
    Using.resource(connectorInitializer.init(settings)) { scmConnector =>
      importCrawler.findAllImports(settings.getScmPath)(scmConnector, version).toList
        .sortWith(Sorting.byDepthAndAlphabetically)
    }
  }

  def getVersionableFileNames(folderId: String): List[String] = wrapExceptions {
    val versionSettingsConfig = getSettings(folderId)
    versioningStrategyResolver.resolve(versionSettingsConfig.getVersioningStyle).getVersionableFileNames(folderId, versionSettingsConfig)
      .sortWith(Sorting.byDepthAndAlphabetically)
  }

  def getValidationMessage(messageId: Integer): ValidationReportMessage = {
    folderValidationMessageRepository.findValidationMessageById(messageId).orNull
  }

  def getValidationMessages(folderId: String): ValidationReport = {
    ValidationReport.createWarningReport(folderValidationMessageRepository.findValidationMessagesByFolderId(folderId).asJava)
  }

  def setValidationMessages(folderId: String, validationMessages: Seq[ValidationReportMessage]): Unit = {
    folderValidationMessageRepository.findValidationMessagesByFolderId(folderId).foreach(message => folderValidationMessageRepository.deleteValidationMessage(message.getMessageId))
    validationMessages.foreach(message => folderValidationMessageRepository.createValidationMessageEntity(folderId, message))
  }

  def updateValidationMessageStatus(messageId: Integer, validationMessage: ValidationReportMessage): Unit = {
    folderValidationMessageRepository.updateValidationMessageResolvedState(messageId, validationMessage)
  }

  def getVersioningStyles: List[VersioningStyle] = versioningStrategyResolver.supportedStyles

  private def populateCommitUrls(gitConnection: Repository, versions: Seq[VersionInfo]): Unit = {
    val provider = Option(gitConnection).map(_.getGitProvider)
    if (gitConnection != null && provider.nonEmpty) {
      val baseUrl = gitConnection.getUrl
      versions.foreach { version =>
        version.setCommitUrl(generateCommitUrl(provider.get, baseUrl, version.getCommitHash).orNull)
      }
    }
  }

  private def generateCommitUrl(gitProvider: GitProvider, baseUrl: String, commitHash: String): Option[String] = {
    if (gitProvider == null || baseUrl == null || commitHash == null) {
      None
    } else {
      val cleanBaseUrl = baseUrl.stripSuffix("/").stripSuffix(".git")
      gitProvider match {
        case GitProvider.GITHUB => Some(s"$cleanBaseUrl/commit/$commitHash")
        case GitProvider.GITLAB => Some(s"$cleanBaseUrl/-/commit/$commitHash")
        case GitProvider.BITBUCKET => Some(s"$cleanBaseUrl/commits/$commitHash")
        case GitProvider.AZURE => Some(s"$cleanBaseUrl/commit/$commitHash")
      }
    }
  }

  private def disableTriggers(folderId: String): Seq[Trigger] = {
    logger.warn(s"Disabling all enabled triggers in folder $folderId before changing folder contents")
    val triggers = triggerRepository.findByFolderId(folderId, nestedFolders = true, PageRequest.of(0, Int.MaxValue))
      .getContent
      .asScala
      .filter(trigger => trigger.isEnabled)
      .toSeq


    triggers.foreach(trigger => {
      logger.warn(s"Disabling trigger title=[${trigger.getTitle}] id=[${trigger.getId}]")
      triggerService.updateTriggerStatus(trigger.getId, enabled = false, checkReferencePermissions = false)
    })

    triggers
  }

  private[scm] def applyDefinitionsInTransaction(includes: ImportIncludes,
                                                 folderId: String,
                                                 definitions: Seq[Definition],
                                                 scmData: Option[SCMTraceabilityData] = None,
                                                 gitRepo: Option[Repository] = None,
                                                 validator: Option[ImportValidator] = None): ImportResult = {


    val importResult = executeTransaction(includes, folderId, definitions, scmData, gitRepo, validator)
    importResult.postCommitActions.foreach(action => action.run()) // TODO: should we wrap the run calls in try/catch log failures and continue?
    importResult
  }

  //noinspection ScalaStyle
  private def executeTransaction(includes: ImportIncludes,
                                 folderId: String,
                                 definitions: Seq[Definition],
                                 scmData: Option[SCMTraceabilityData] = None,
                                 gitRepo: Option[Repository],
                                 validator: Option[ImportValidator] = None): ImportResult = {

    val disabledTriggers = if (includes.triggers) disableTriggers(folderId) else Seq.empty[Trigger]

    try {
      transactionTemplate.execute { _ =>
        // make sure Environments are processed first as other definitions could depend on it existing
        val ir = definitions.sortBy(_.kind != "Environments").foldLeft(ImportResult.empty) { case (acc, definition) =>
          val spec = gitRepo match {
            case Some(repo) =>
              definition.spec match {
                case ciSpec: CiSpec =>
                  val cisExcludingVersioningGitRepo = ciSpec.cis.filterNot(config => config.isInstanceOf[Repository]
                    && config.asInstanceOf[Repository].getTitle.equals(repo.getTitle))
                  CiSpec(cisExcludingVersioningGitRepo, ciSpec.references, ciSpec.files)
                case _ =>
                  definition.spec
              }
            case None => definition.spec
          }

          val updatedDefinition = Definition(definition.apiVersion, createMetadata(folderId, definition.metadata), definition.kind, spec)
          acc.merge(importService.process(includes, updatedDefinition, scmData, validator))
        }
        val gitRepoId = gitRepo match {
          case Some(repo) => Some(repo.getId)
          case None => None
        }
        val processedCis = orphanDeletionService.processCis(includes, folderId, ir.changedIdsPerTypePerFolder, gitRepoId)
        val processedPermissions = orphanDeletionService.processPermissions(includes, folderId, ir.changedIds)

        val fullIr = ir.merge(processedCis).merge(processedPermissions)
        fullIr
      }
    } catch {
      case e: Throwable =>
        // Transaction was rolled back, re-enable the triggers that were disabled before starting the transaction
        logger.warn(s"Apply failed and changes were rolled back in $folderId re-enabling the triggers that were disabled before attempting changes")

        disabledTriggers.foreach(trigger => {
          logger.warn(s"Enabling trigger title=[${trigger.getTitle}] id=[${trigger.getId}]")
          triggerService.updateTriggerStatus(trigger.getId, enabled = true, checkReferencePermissions = false)
        })
        throw e
    }
  }

  private def getFolderSecrets(folderId: String, settings: FolderVersioningSettings): Map[String, String] = {
    try {
      val definitions = definitionsGenerator.generate(folderId, settings)
      val stream = new ByteArrayOutputStream()
      val generateCtxs = XLRDefinitionWriter().writeYaml(stream, WriterConfig(definitions, writeSecrets = false, writeDefaults = false))
      generateCtxs.flatMap(_.secrets).toMap
    } catch {
      case e: Throwable =>
        if (!e.getMessage.equals(EMPTY_DEFINITION_ERROR)) {
          logger.warn(s"Can't generate YAML for current state of folder $folderId: ${e.getMessage}")
        }
        Map()
    }
  }

  private def validateSettings(config: FolderVersioningSettings, remoteValidations: Boolean): Unit = {
    folderVersioningConfigService.validateConfigData(config, resolveFolderTitleFromId(config.getFolderId))
    Using.resource(connectorInitializer.init(config)) { connector =>
      val url = connector.getConnectionSettings.getUrl
      if (!isFileRepo(url)) {
        verifyRepositoryPath(url)
        if (remoteValidations) {
          connector.verifyRemoteBranch(url, config.getBranch, Some(config.getGitConnection.getTitle))
        }
      }
    }
  }

  private def createMetadata(folderId: String, metadataOption: Option[Map[String, String]]): Option[Map[String, String]] = {
    val metadata = metadataOption.getOrElse(Map.empty)
    val versionedFolderPath = folderAsCodeService.getFolderPath(folderId)
    val metadataHome = if (metadata.contains(MetadataFields.PATH.toString)) {
      s"$versionedFolderPath/${metadata(MetadataFields.PATH.toString)}".replace("//", "/").stripSuffix("/")
    } else {
      versionedFolderPath
    }

    Some(Map(
      MetadataFields.FOLDER.toString -> folderId,
      MetadataFields.HOMEFOLDER.toString -> metadataHome
    ))
  }

  private def markAsApplied(version: String, config: FolderVersioningSettings): Unit = {
    val now = new Date
    config.setLastFetched(now)
    config.setAppliedVersion(version)
    config.setAppliedDate(now)
    config.setAppliedBy(getAuthenticatedUserName)
  }

  private def applyScmDataToDefinitions(definitions: Seq[Definition], scmData: SCMTraceabilityData): Unit = {
    def flattenCis(cis: Seq[ConfigurationItem]): Seq[ConfigurationItem] = cis.flatMap {
      case folder: Folder => flattenCis(folder.getChildren.asScala.toSeq)
      case ci => Seq(ci)
    }

    val foldersAndCis = definitions.withFilter(_.spec.isInstanceOf[CiSpec]).flatMap(_.spec.asInstanceOf[CiSpec].cis)
    val justCis = flattenCis(foldersAndCis)
    justCis.foreach {
      // only templates support scmData currently
      case t: Release => releaseService.createSCMData(t.getId, scmData)
      case _ => ()
    }
  }

  private def getScmTraceabilityData(config: FolderVersioningSettings, versionInfo: VersionInfo): SCMTraceabilityData = {
    new SCMTraceabilityData(
      "git",
      s"${versionInfo.getCommitHash}",
      versionInfo.getAuthor,
      new DateTime(versionInfo.getCommitTime),
      s"[${versionInfo.getName}] ${versionInfo.getShortMessage}",
      config.getGitConnection.getUrl,
      getDefinitionsPath(config.getScmPath))
  }

  private def resolveFolderTitleFromId(folderId: String): String = {
    Try(folderService.findById(folderId))
      .map(_.getTitle)
      .getOrElse("")
  }

}

object Sorting {
  def byDepthAndAlphabetically(first: String, second: String): Boolean = {
    val firstTokens = first.split("/")
    val secondTokens = second.split("/")

    if (firstTokens.length < secondTokens.length) {
      true
    } else if (firstTokens.length > secondTokens.length) {
      false
    } else {
      var res = true
      breakable {
        for (i <- 0 until firstTokens.length) {
          val comparisonResult = firstTokens(i).compareTo(secondTokens(i))
          if (comparisonResult != 0) {
            res = comparisonResult < 0
            break
          }
        }
      }
      res
    }
  }
}
