package com.xebialabs.xlrelease.versioning.ascode.scm.strategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.ascode.yaml.writer.AdditionalFile
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, generateImportsSpecDefinition}
import com.xebialabs.xlrelease.ascode.service.{FolderAsCodeService, FolderSearch}
import com.xebialabs.xlrelease.ascode.yaml.writer.XLRDefinitionWriter
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettingsUtil.getDefinitionsPath
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.{FolderVersioningSettings, FolderVersioningSettingsUtil}
import com.xebialabs.xlrelease.scm.connector.{BinaryFile, ScmBlobs}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.GENERATE_FOLDER_CONFIGURATION
import com.xebialabs.xlrelease.service.{FolderService, WorkdirSupport}
import org.apache.commons.io.IOUtils

import java.io.ByteArrayOutputStream
import scala.jdk.CollectionConverters.SetHasAsScala
import scala.util.Using

trait BaseVersioningStrategy extends VersioningStyleStrategy with WorkdirSupport {

  val HOMEFOLDER: String = MetadataFields.HOMEFOLDER.toString
  val PATH: String = MetadataFields.PATH.toString

  def getFolderService: FolderService

  def getFolderAsCodeService: FolderAsCodeService

  def getPermissions: PermissionChecker

  def performGenerate(scope: FolderSearch, rootPath: String, settings: FolderVersioningSettings): Map[String, List[Definition]]

  def isMultiFile: Boolean

  override def generateFolder(folderId: String, settings: FolderVersioningSettings): FolderContent = {
    try {
      val definitions = generateDefinitions(folderId, settings)
      val folders = generateContent(definitions)

      val allBinaryFiles = folders.foldLeft(Seq[BinaryFile]())((acc, folder) => acc ++ folder.blob.filesToAdd)
      val allSecrets = folders.foldLeft(Map[String, String]())((acc, folder) => acc ++ folder.secrets)
      val allFileDefinitions = folders.foldLeft(List[Definition]())((acc, folder) => acc ++ folder.definitions)

      FolderContent(ScmBlobs(allBinaryFiles), allFileDefinitions, allSecrets)
    } catch {
      case e: AsCodeException => throw ScmException(e.getMessage, e.getCause)
    }
  }

  private def generateContent(definitions: Map[String, List[Definition]]) = {
    definitions.foldLeft(List[FolderContent]())((accumulator, fileDefinitions) => {
      val stream = new ByteArrayOutputStream()
      val generateCtxs = XLRDefinitionWriter().writeYaml(stream, WriterConfig(fileDefinitions._2, writeSecrets = false, writeDefaults = false))
      val secrets = generateCtxs.flatMap(_.secrets).toMap
      val definitionAbsolutePath = fileDefinitions._1.substring(0, fileDefinitions._1.lastIndexOf("/"))
      val additionalFiles = generateCtxs.flatMap(_.additionalFile).map(toBinaryFile(_, definitionAbsolutePath))
      val filesToAdd = Seq(BinaryFile(fileDefinitions._1, () => stream.toByteArray)) ++ additionalFiles
      accumulator :+ FolderContent(
        ScmBlobs(filesToAdd),
        fileDefinitions._2,
        secrets
      )
    })
  }

  private def toBinaryFile(af: AdditionalFile, definitionPath: String) = {
    BinaryFile(
      FolderVersioningSettingsUtil.getDefinitionsPath(definitionPath, af.name),
      () => {
        withWorkDir {
          Using.resource(af.file.getInputStream) { is =>
            IOUtils.toByteArray(is)
          }
        }
      })
  }

  def generateDefinitions(folderId: String, settings: FolderVersioningSettings): Map[String, List[Definition]] = {
    val folder = getFolderService.findViewableFoldersById(folderId)
    val rootPath = getFolderAsCodeService.getFolderPath(folderId)
    val entries = traverseFolder(folder, None, rootPath, settings)
    entries.map(entry => createAbsoluteScmPath(entry._1, settings.getScmPath) -> entry._2)
  }

  private def createAbsoluteScmPath(relativePath: String, scmPath: String): String = {
    val relativePathWithoutRootFolderName = relativePath.substring(relativePath.indexOf("/") + 1)
    s"$scmPath/$relativePathWithoutRootFolderName"
  }

  private def traverseFolder(folder: Folder, parentPath: Option[String], rootPath: String,
                             settings: FolderVersioningSettings): Map[String, List[Definition]] = {
    val folderPath = createFolderPath(folder, parentPath)
    val scope = FolderSearch(folderPath, folder.getId, includeSubFolders = false)
    val currentFolderDefs = performGenerate(scope, rootPath, settings)

    val childFolders = folder.getChildren.asScala.toSet
    val childrenDefs = childFolders.flatMap(traverseFolder(_, Some(folderPath), rootPath, settings)).toMap

    val releasefileDefs = createReleasefileDefinitions(folderPath, currentFolderDefs.keys.toList)

    currentFolderDefs ++ childrenDefs ++ releasefileDefs
  }

  private def createFolderPath(folder: Folder, parentPath: Option[String]) = {
    parentPath match {
      case None => folder.getTitle
      case Some(path) => s"$path/${folder.getTitle}"
    }
  }

  private def createReleasefileDefinitions(folderPath: String, keys: List[String]) = {
    if (isMultiFile) {
      val imports = keys.map(FolderVersioningSettingsUtil.getfilename)
      val importsDefinition = if (imports.isEmpty) None else Some(generateImportsSpecDefinition(imports))
      Map(getDefinitionsPath(folderPath) -> importsDefinition.toList)
    } else {
      Map.empty
    }
  }

  override def getVersionableFileNames(folderId: String, settings: FolderVersioningSettings): List[String] = {
    val definitions = generateDefinitions(folderId, settings)
    definitions.keys.map(_.substring(settings.getScmPath.length + 1)).toList
  }

  private def validateGeneratePermissions(folderId: String): Unit = {
    if (!(getPermissions.hasPermission(GENERATE_FOLDER_CONFIGURATION, folderId) || getPermissions.isCurrentUserAdmin)) {
      throw ScmException(s"Unable to generate configuration: must have Admin or 'Generate Configuration' permission on the folder")
    }
  }

  implicit class DefinitionOps(definition: Definition) {
    def adjustMetadata(rootPath: String): Definition = {
      val lastPartOfRootPath = if (rootPath.contains("/")) {
        rootPath.substring(rootPath.lastIndexOf("/") + 1)
      } else {
        rootPath
      }

      val path = definition.metadata.get(HOMEFOLDER).strip(lastPartOfRootPath)
      val home = s"$rootPath${if (path.equals("/")) "" else s"/$path"}"
      val updatedMetadata = Map(PATH -> path, HOMEFOLDER -> home)
      definition.copy(metadata = Some(updatedMetadata))
    }

    implicit class StringOps(path: String) {
      def strip(prefix: String): String = {
        val fixed = path.stripPrefix(prefix).stripPrefix("/")
        if (fixed.isEmpty) {
          "/" // empty string would be the best but jackson will treat a map with one key and empty string value as empty map
        } else {
          fixed
        }
      }
    }
  }

  implicit class FolderVersioningSettingsOps(settings: FolderVersioningSettings) {
    def splitAndConvert(): Seq[CisConfig] =
      Seq(CisConfig(generateConfigurations = settings.getExportConfiguration),
        CisConfig(generateDashboards = settings.getExportDashboards),
        CisConfig(generateTemplates = settings.getExportTemplates),
        CisConfig(generateWorkflows = settings.getExportWorkflows),
        CisConfig(generateTriggers = settings.getExportTriggers),
        CisConfig(generateVariables = settings.getExportVariables),
        CisConfig(generateDeliveryPatterns = settings.getExportPatterns),
        CisConfig(generateNotificationSettings = settings.getExportNotifications),
        CisConfig(generatePermissions = settings.getExportSecurity),
        CisConfig(generateApplications = settings.getExportApplications),
        CisConfig(generateEnvironments = settings.getExportEnvironments)).filter(!_.isEmpty())

    def toCisConfig: CisConfig = CisConfig(
      generateConfigurations = settings.getExportConfiguration,
      generateDashboards = settings.getExportDashboards,
      generateTemplates = settings.getExportTemplates,
      generateWorkflows = settings.getExportWorkflows,
      generateTriggers = settings.getExportTriggers,
      generateVariables = settings.getExportVariables,
      generateDeliveryPatterns = settings.getExportPatterns,
      generateNotificationSettings = settings.getExportNotifications,
      generatePermissions = settings.getExportSecurity,
      generateApplications = settings.getExportApplications,
      generateEnvironments = settings.getExportEnvironments
    )
  }
}
