package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.xlrelease.api.v1.EnvironmentReservationApi
import com.xebialabs.xlrelease.ascode.utils.ImportContext
import com.xebialabs.xlrelease.domain.environments.EnvironmentReservation
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.ValidationMessage
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentRepository}
import com.xebialabs.xltype.serialization.CiReference
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class EnvironmentReservationAsCodeService(environmentReservationApi: EnvironmentReservationApi,
                                          environmentRepository: EnvironmentRepository,
                                          applicationRepository: ApplicationRepository) {

  def process(context: ImportContext, environmentReservation: EnvironmentReservation): ImportResult = {
    create(context, environmentReservation)
  }

  private def create(context: ImportContext, environmentReservation: EnvironmentReservation) = {
    resolveReferences(context.references)
    val messages = validate(context, environmentReservation)
    val created = environmentReservationApi.create(environmentReservation)
    ImportResult(List(CI.ids.withCreated(created.getId)),
      Seq.empty,
      Map.empty,
      messages
    )
  }

  private def validate(context: ImportContext, environmentReservation: EnvironmentReservation): List[ValidationMessage] = {
    context.validator match {
      case Some(validator) => validator.validateCi(environmentReservation, context.getFolderInfo()).toList
      case None => List.empty
    }
  }

  private def resolveReferences(references: List[CiReference]): Unit = {
    references.foreach(reference => {
      reference.getProperty.getName match {
        case "environment" =>
          val environment = reference.getIds.asScala.map(ref => {
            environmentRepository.findEnvironmentByTitleAndFolder(ref, None)
              .getOrElse(environmentRepository.findEnvironmentById(ref))
          }).toList.head
          reference.getCi.setProperty("environment", environment)
        case "applications" =>
          val applications = reference.getIds.asScala.map(ref => {
            applicationRepository.findApplicationByTitleAndFolder(ref, None)
              .getOrElse(applicationRepository.findApplicationById(ref))
          }).toList.asJava
          reference.getCi.setProperty("applications", applications)
      }
    })
  }

}
