package com.xebialabs.xlrelease.api.v1.impl

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.api.v1.FolderVersioningApi
import com.xebialabs.xlrelease.api.v1.form.CreateVersionForm
import com.xebialabs.xlrelease.api.v1.views.{VersionInfo, VersionsView}
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.{ValidationReport, ValidationReportMessage}
import com.xebialabs.xlrelease.domain.versioning.ascode.{GitBranch, VersioningStyle}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.{APPLY_FOLDER_CHANGES, EDIT_FOLDER_VERSIONS, GENERATE_FOLDER_CONFIGURATION, VIEW_FOLDER_VERSIONS}
import com.xebialabs.xlrelease.service.{ConfigurationService, FolderService}
import com.xebialabs.xlrelease.versioning.ascode.actors.FolderVersioningActorService
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.VersioningStrategyResolver
import com.xebialabs.xlrelease.versioning.ascode.scm.{FolderVersioningPreviewService, FolderVersioningService}
import jakarta.ws.rs.PathParam
import org.springframework.stereotype.Controller

import java.{lang, util}
import scala.jdk.CollectionConverters._

@Controller
class FolderVersioningApiImpl(permissionChecker: PermissionChecker,
                              folderVersioningService: FolderVersioningService,
                              previewService: FolderVersioningPreviewService,
                              folderVersioningActorService: FolderVersioningActorService,
                              versioningStyleResolver: VersioningStrategyResolver,
                              configurationService: ConfigurationService,
                              folderService: FolderService) extends FolderVersioningApi {

  override def getVersions(folderId: String, fetchChanges: lang.Boolean): VersionsView = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    val (lastFetched, versions) = if (fetchChanges) {
      folderVersioningActorService.fetchChanges(folderId)
    } else {
      folderVersioningService.getVersions(folderId)
    }
    new VersionsView(lastFetched, versions.asJava)
  }

  override def getVersions(folderId: String): VersionsView = {
    getVersions(folderId, false)
  }

  override def getAllBranches(folderId: String, configId: String): java.util.List[GitBranch] = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.getAllBranches(configId).asJava
  }

  override def createVersion(folderId: String, versionForm: CreateVersionForm): VersionInfo = {
    permissionChecker.check(GENERATE_FOLDER_CONFIGURATION, folderId)
    checkArgument(Option(versionForm.getName).exists(_.trim.nonEmpty), "Version name must be provided.")
    folderVersioningActorService.createVersion(folderId, versionForm.getName, Option(versionForm.getDescription).getOrElse(""))
  }

  override def createVersion(folderId: String, versionName: String, description: String): VersionInfo = {
    val createVersionForm = new CreateVersionForm
    createVersionForm.setName(versionName)
    createVersionForm.setDescription(description)
    createVersion(folderId, createVersionForm)
  }

  override def applyVersion(folderId: String, version: String): ValidationReport = {
    // TODO: which permission to check here?
    permissionChecker.checkAny(folderId, APPLY_FOLDER_CHANGES)
    folderVersioningService.applyVersion(folderId, version)
  }

  override def previewCurrent(folderId: String, filename: String): String = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    previewService.generatePreview(folderId, None, Option(filename).getOrElse(""))
  }

  override def previewVersion(folderId: String, version: String, filename: String): String = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    previewService.generatePreview(folderId, Some(version), filename)
  }

  override def getVersionedFileNames(folderId: String, version: String): util.List[String] = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.getVersionedFileNames(folderId, version).asJava
  }

  override def getCurrentVersionableFileNames(folderId: String): util.List[String] = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.getVersionableFileNames(folderId).asJava
  }

  override def getSettings(@PathParam("folderId") folderId: String): FolderVersioningSettings = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.findSettings(folderId).orNull
  }

  override def updateSettings(@PathParam("folderId") folderId: String, config: FolderVersioningSettings): FolderVersioningSettings = {
    permissionChecker.check(EDIT_FOLDER_VERSIONS, folderId)
    validateFolderVersioningSettings(config)
    config.setFolderId(folderId)
    try {
      folderVersioningActorService.createOrUpdateSettings(config)
    } catch {
      case e: ScmException => throw new IllegalArgumentException(s"Unable to update folder versioning settings: ${e.getMessage}", e)
    }
  }

  override def deleteSettings(@PathParam("folderId") folderId: String): Unit = {
    permissionChecker.check(ADMIN)
    checkArgument(folderService.exists(folderId), s"Folder '$folderId' does not exist in the repository.")
    folderVersioningActorService.cleanLocalRepo(folderId, clusterWide = true)
    folderVersioningService.deleteSettings(folderId)
  }

  override def deleteLocalRepo(folderId: String, clusterWide: lang.Boolean): Unit = {
    permissionChecker.check(ADMIN)
    folderVersioningActorService.cleanLocalRepo(folderId, clusterWide)
  }

  override def deleteLocalRepo(folderId: String): Unit = {
    deleteLocalRepo(folderId, false)
  }

  override def resetLocalRepo(folderId: String, clusterWide: lang.Boolean): Unit = {
    permissionChecker.check(ADMIN)
    val config = folderVersioningService.getSettings(folderId)
    folderVersioningActorService.resetLocalRepo(config, clusterWide)
  }

  override def validateCurrent(folderId: String): ValidationReport = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.validate(folderId)
  }

  override def getValidationMessages(folderId: String): ValidationReport = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.getValidationMessages(folderId)
  }

  override def setValidationMessages(folderId: String, validationReport: ValidationReport): Unit = {
    permissionChecker.check(EDIT_FOLDER_VERSIONS, folderId)
    folderVersioningService.setValidationMessages(folderId, validationReport.getWarnings.asScala.toSeq)
  }

  override def getValidationMessage(folderId: String, messageId: String): ValidationReportMessage = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    Option(folderVersioningService.getValidationMessage(messageId.toInt))
      .getOrElse(throw new IllegalArgumentException(s"Validation message with id $messageId not found"))
  }

  override def setValidationMessagesStatus(folderId: String, messageId: String, validationMessage: ValidationReportMessage): Unit = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    folderVersioningService.updateValidationMessageStatus(messageId.toInt, validationMessage)
  }

  override def clearSecrets(folderId: String): Unit = {
    permissionChecker.check(ADMIN)
    folderVersioningActorService.clearSecrets(folderId)
  }

  override def getVersioningStyles: util.List[VersioningStyle] = {
    folderVersioningService.getVersioningStyles.asJava
  }

  override def compareVersions(folderId: String, fileName: String, leftVersion: String, rightVersion: String): String = {
    permissionChecker.check(VIEW_FOLDER_VERSIONS, folderId)
    //Validate input
    checkArgument(fileName != null && !fileName.isEmpty, "File name must be provided for comparison.")
    folderVersioningService.compareVersions(folderId, fileName, leftVersion, rightVersion)
  }

  override def compareWithLatest(folderId: String, fileName: String, version: String): String = {
    compareVersions(folderId, fileName, null, version)
  }

  private def validateFolderVersioningSettings(config: FolderVersioningSettings): Unit = {
    validateVersioningStyle(config)
    validateGitConnection(config)
  }

  private def validateVersioningStyle(config: FolderVersioningSettings): Unit = {
    val styleInput = Option(config.getVersioningStyle).map(_.trim)

    styleInput match {
      case None =>
        // No versioning style provided → nothing to validate
        return

      case Some(style) if style.nonEmpty =>
        val supported = versioningStyleResolver.supportedStyles
        val canonical = supported
          .find(s => s.id.equalsIgnoreCase(style))
          .map(_.id)
          .getOrElse {
            val allowed = supported.map(_.id).mkString(", ")
            throw new IllegalArgumentException(
              s"Invalid versioningStyle '$style'. Supported versioning styles: $allowed"
            )
          }

        // Normalize to canonical id
        config.setVersioningStyle(canonical)

      case Some(_) =>
        val allowed = versioningStyleResolver.supportedStyles.map(_.id).mkString(", ")
        throw new IllegalArgumentException(
          s"Missing versioningStyle. Supported versioning styles: $allowed"
        )
    }
  }

  private def validateGitConnection(config: FolderVersioningSettings): Unit = {
    val gitConnectionValid = Option(config.getGitConnection)
      .exists(conn => configurationService.exists(conn.getId))

    checkArgument(gitConnectionValid, "A valid gitConnection must be specified.")
  }
}
