package com.xebialabs.xlrelease.versioning.ascode.repository

import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.{ExternalDependencyReportMessage, ValidationReportMessage}
import com.xebialabs.xlrelease.versioning.ascode.repository.sql.persistence.{FolderValidationMessagePersistence, FolderValidationMessageRow}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository

@IsTransactional
@Repository
@Autowired
class SqlFolderValidationMessageRepository(folderValidationMessagePersistence: FolderValidationMessagePersistence
                                          ) extends FolderValidationMessageRepository with Logging {

  @Timed
  @IsReadOnly
  override def findValidationMessageById(messageId: Integer): Option[ValidationReportMessage] = {
    logger.debug(s"Getting folder validation message with id [$messageId]")
    folderValidationMessagePersistence.findById(messageId).map(folderValidationMessageRowToReportMessage)
  }

  @Timed
  @IsReadOnly
  override def findValidationMessagesByFolderId(folderId: String): Seq[ValidationReportMessage] = {
    logger.debug(s"Getting validation message entries for folderId [$folderId]")
    folderValidationMessagePersistence.findByFolderId(folderId).map(folderValidationMessageRowToReportMessage)
  }

  @Timed
  override def updateValidationMessageResolvedState(messageId: Integer, validationMessage: ValidationReportMessage): Unit = {
    logger.debug(s"Updating folder validation message with id [$messageId] with resolvedStatus [$validationMessage.resolvedStatus]")
    folderValidationMessagePersistence.update(messageId, validationMessage)
  }

  @Timed
  override def createValidationMessageEntity(folderId: String, validationMessage: ValidationReportMessage): Unit = {
    logger.debug(s"Creating validation message entry for folderId [$folderId] with resolvedStatus [${validationMessage.getResolvedStatus}] and message [$validationMessage]")
    folderValidationMessagePersistence.insert(folderId, validationMessage)
  }

  @Timed
  override def deleteValidationMessage(messageId: Integer): Unit = {
    logger.debug(s"Deleting folder validation message with id [$messageId]")
    folderValidationMessagePersistence.delete(messageId)
  }

  private def folderValidationMessageRowToReportMessage(row: FolderValidationMessageRow): ExternalDependencyReportMessage = {
    val message = new ExternalDependencyReportMessage
    message.setId(row.ciInfoId)
    message.setCiType(row.ciInfoType)
    message.setCiTypeDisplayName(row.ciInfoDisplayName)
    message.setTitle(row.ciInfoTitle)
    message.setFolder(row.ciInfoFolderPath)
    message.setFolderId(row.ciInfoFolderId)
    message.setProperty(row.property)
    message.setMessage(row.message)
    message.setMessageType(row.messageType)
    message.setMessageId(row.messageId)
    message.setResolvedStatus(row.resolvedStatus)
    message.setDependency(row.dependency)
    message
  }

}
