package com.xebialabs.xlrelease.versioning.ascode.scm

import com.xebialabs.ascode.yaml.model.permission.PermissionsSpec
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.yaml.model.ImportsSpec
import com.xebialabs.xlrelease.ascode.yaml.parser.XLRDefinitionParser
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettingsUtil.FOLDER_VERSIONING_YAML_FILENAME
import com.xebialabs.xlrelease.scm.connector.BinaryFile
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.AsCodeJGitConnector
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import org.yaml.snakeyaml.parser.ParserException
import org.yaml.snakeyaml.scanner.ScannerException

import java.nio.charset.StandardCharsets

/**
 * 2 kinds of definitions are supported by DefinitionCrawler. Or, to be specific, 2 metadata home data points are supported: "metadata/home" and "metadata/path"
 * - "home" is considered to be an absolute path without the starting slash, and is created by system versions 23.3.x and prior.
 * - "path" is introduced in 24.1.0 and is a relative path.
 *
 * Definition crawler WILL always return absolute paths, calculation of which will be based upon the path of the folder that is versioned, and the name of the
 * home path in the definition that is read from the scm files. Different strategies will be used depending on what's found in the yaml, but the outputed
 * metadata will always be "metadata/home" because that's what CiSpecInterpreter and PermissionsSpecInterpreter work with.
 */
@Component
class DefinitionCrawler @Autowired()(definitionParser: XLRDefinitionParser) {

  private val HOMEFOLDER: String = MetadataFields.HOMEFOLDER.toString

  def findAllDefinitions(scmPath: String, git: AsCodeJGitConnector, version: String): List[Definition] = {
    val releasefiles = git.checkoutByPathAndFilenameRecursive(scmPath, FOLDER_VERSIONING_YAML_FILENAME, version)
    releasefiles.filesToAdd.flatMap(processReleasefile(_, scmPath, git, version)).toList
  }

  private def processReleasefile(releasefile: BinaryFile,
                                 scmPath: String,
                                 git: AsCodeJGitConnector,
                                 version: String): List[Definition] = {
    val defs = getDefinitionsFromFile(releasefile)
    val defsWithPaths = collectImportedDefinitions(defs, git, version)
    defsWithPaths.map(_.fixFilePaths(scmPath))
  }

  private def collectImportedDefinitions(defs: List[DefWithPath], git: AsCodeJGitConnector, version: String): List[DefWithPath] = {
    defs.foldLeft(List[DefWithPath]()) { (acc, dwp) =>
      dwp.definition.spec match {
        case _: ImportsSpec => acc ++ extractImports(dwp, git, version)
        case _ => acc :+ dwp
      }
    }
  }

  private def extractImports(dwp: DefWithPath, scm: AsCodeJGitConnector, version: String): List[DefWithPath] = {
    val absPathWithoutFilename = dwp.definition.metadata.get(HOMEFOLDER)
    val imports = dwp.definition.spec.asInstanceOf[ImportsSpec].imports
    val paths = imports.map(im => s"$absPathWithoutFilename/$im")

    val blobs = scm.checkout(paths.toArray, version).filesToAdd

    blobs.flatMap(getDefinitionsFromFile).toList
  }

  private def getDefinitionsFromFile(blob: BinaryFile) = {
    val yaml = new String(blob.getContent(), StandardCharsets.UTF_8)
    val specsAsStrings = try {
      YamlUtils.preprocessYaml(yaml)
    } catch {
      case e @ (_: ParserException | _: ScannerException) =>
        throw YamlParserException(s"An error occurred while applying the version because the YAML file ${blob.absolutePath} is invalid. Details:", e)
    }
    specsAsStrings.map(definitionParser.parse).map(definition => {
      val definitionWithUpdatedMetadata = definition.spec match {
        case spec if spec.isInstanceOf[CiSpec] || spec.isInstanceOf[PermissionsSpec] => definition
        case spec if spec.isInstanceOf[ImportsSpec] => definition.copy(metadata = Some(Map(HOMEFOLDER -> blob.absolutePath.withoutFilename)))
        case _ => definition
      }
      DefWithPath(definitionWithUpdatedMetadata, blob.absolutePath)
    })
  }

  implicit class StringOps(string: String) {
    def withoutFilename: String = string.substring(0, string.lastIndexOf("/"))
  }

}

case class DefWithPath(definition: Definition, absolutePath: String) {
  def fixFilePaths(scmPath: String): Definition = {
    val noScm = absolutePath.stripPrefix(s"$scmPath/")
    val folder = noScm.substring(0, noScm.lastIndexOf("/") + 1) // strip filename from a yaml containing this definition
    definition.spec match {
      case spec: CiSpec =>
        val files = spec.files.map(e => e._1 -> s"$folder${e._2}")
        definition.copy(spec = spec.copy(files = files))
      case _ => definition
    }
  }
}